<?php

namespace Drupal\Tests\content_translation_access\Kernel;

use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\content_translation_access\Permissions;

/**
 * Tests dynamic permission generation.
 *
 * @coversDefaultClass \Drupal\content_translation_access\Permissions
 *
 * @group content_translation_access
 */
class DynamicPermissionsTest extends ContentTranslationAccessKernelTestBase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create additional content types.
    $this->createContentType(['type' => 'article', 'name' => 'Article']);
    $this->createContentType(['type' => 'news', 'name' => 'News']);

    // Enable content translation for article.
    $this->container->get('content_translation.manager')
      ->setEnabled('node', 'article', TRUE);
  }

  /**
   * Test dynamic permission generation for content types.
   *
   * @covers ::ctaPermissions
   * @covers ::buildPermissions
   */
  public function testDynamicPermissionGeneration() {
    // Create the permissions object.
    $permissions_generator = Permissions::create($this->container);

    // Get generated permissions.
    $permissions = $permissions_generator->ctaPermissions();

    // Assert that permissions are generated for enabled bundles.
    // Page content type (enabled in parent setUp).
    $this->assertArrayHasKey('cta create translation node page', $permissions,
      'Create translation permission should exist for page.');
    $this->assertArrayHasKey('cta translate node page', $permissions,
      'Translate permission should exist for page.');
    $this->assertArrayHasKey('cta delete translation node page', $permissions,
      'Delete translation permission should exist for page.');

    // Article content type (enabled in this test).
    $this->assertArrayHasKey('cta create translation node article', $permissions,
      'Create translation permission should exist for article.');
    $this->assertArrayHasKey('cta translate node article', $permissions,
      'Translate permission should exist for article.');
    $this->assertArrayHasKey('cta delete translation node article', $permissions,
      'Delete translation permission should exist for article.');

    // News content type (NOT enabled for translation).
    $this->assertArrayNotHasKey('cta create translation node news', $permissions,
      'Permissions should not exist for non-translatable content type.');
    $this->assertArrayNotHasKey('cta translate node news', $permissions,
      'Permissions should not exist for non-translatable content type.');
    $this->assertArrayNotHasKey('cta delete translation node news', $permissions,
      'Permissions should not exist for non-translatable content type.');

    // Check for the general permission.
    $this->assertArrayHasKey('cta translate any entity', $permissions,
      'General translate any entity permission should exist.');

    // Verify permission structure.
    $page_create_permission = $permissions['cta create translation node page'];
    $this->assertArrayHasKey('title', $page_create_permission,
      'Permission should have a title.');
    $this->assertStringContainsString('Create', (string) $page_create_permission['title'],
      'Create permission title should contain "Create".');
    $this->assertStringContainsString('Page', (string) $page_create_permission['title'],
      'Permission title should contain bundle label.');
  }

  /**
   * Test permission generation for multiple entity types.
   *
   * @covers ::ctaPermissions
   */
  public function testPermissionGenerationForMultipleEntityTypes() {
    // Enable taxonomy module and create a vocabulary.
    $this->enableModules(['taxonomy']);
    $this->installEntitySchema('taxonomy_term');
    $this->installConfig(['taxonomy']);

    // Create a vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
    ]);
    $vocabulary->save();

    // Enable content translation for taxonomy terms.
    $this->container->get('content_translation.manager')
      ->setEnabled('taxonomy_term', 'tags', TRUE);

    // Create the permissions object.
    $permissions_generator = Permissions::create($this->container);

    // Get generated permissions.
    $permissions = $permissions_generator->ctaPermissions();

    // Assert permissions for node.
    $this->assertArrayHasKey('cta translate node page', $permissions,
      'Node permissions should be generated.');

    // Assert permissions for taxonomy_term.
    $this->assertArrayHasKey('cta create translation taxonomy_term tags', $permissions,
      'Taxonomy term create permission should be generated.');
    $this->assertArrayHasKey('cta translate taxonomy_term tags', $permissions,
      'Taxonomy term translate permission should be generated.');
    $this->assertArrayHasKey('cta delete translation taxonomy_term tags', $permissions,
      'Taxonomy term delete permission should be generated.');
  }

  /**
   * Test hasPermission static method.
   *
   * @covers ::hasPermission
   */
  public function testHasPermissionStaticMethod() {
    // Create users with different permissions.
    $user_with_translate = $this->createUser([
      'cta translate node page',
    ]);
    $this->assertNotFalse($user_with_translate, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_translate);

    $user_with_create = $this->createUser([
      'cta create translation node page',
    ]);
    $this->assertNotFalse($user_with_create, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_create);

    $user_with_delete = $this->createUser([
      'cta delete translation node page',
    ]);
    $this->assertNotFalse($user_with_delete, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_delete);

    $user_with_bypass = $this->createUser([
      'bypass node access',
    ]);
    $this->assertNotFalse($user_with_bypass, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_bypass);

    $user_with_any = $this->createUser([
      'cta translate any entity',
    ]);
    $this->assertNotFalse($user_with_any, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_any);

    // Test update operation (maps to translate).
    $this->assertTrue(
      Permissions::hasPermission('update', 'node', 'page', $user_with_translate),
      'User with translate permission should have update access.'
    );

    // Test create operation.
    $this->assertTrue(
      Permissions::hasPermission('create', 'node', 'page', $user_with_create),
      'User with create translation permission should have create access.'
    );

    // Test delete operation.
    $this->assertTrue(
      Permissions::hasPermission('delete', 'node', 'page', $user_with_delete),
      'User with delete translation permission should have delete access.'
    );

    // Test bypass permission.
    $this->assertTrue(
      Permissions::hasPermission('update', 'node', 'page', $user_with_bypass),
      'User with bypass permission should have access.'
    );
    $this->assertTrue(
      Permissions::hasPermission('create', 'node', 'page', $user_with_bypass),
      'User with bypass permission should have access.'
    );
    $this->assertTrue(
      Permissions::hasPermission('delete', 'node', 'page', $user_with_bypass),
      'User with bypass permission should have access.'
    );

    // Test translate any entity permission.
    $this->assertTrue(
      Permissions::hasPermission('update', 'node', 'page', $user_with_any),
      'User with translate any entity permission should have access.'
    );
    $this->assertTrue(
      Permissions::hasPermission('create', 'node', 'article', $user_with_any),
      'User with translate any entity permission should have access to any bundle.'
    );

    // Test user without permissions.
    $user_without_permission = $this->createUser(['access content']);
    $this->assertNotFalse($user_without_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_without_permission);
    $this->assertFalse(
      Permissions::hasPermission('update', 'node', 'page', $user_without_permission),
      'User without permission should not have access.'
    );
  }

  /**
   * Test permission generation after disabling translation.
   *
   * @covers ::ctaPermissions
   */
  public function testPermissionRemovalAfterDisablingTranslation() {
    // Create the permissions object.
    $permissions_generator = Permissions::create($this->container);

    // Get initial permissions.
    $permissions = $permissions_generator->ctaPermissions();

    // Verify page permissions exist.
    $this->assertArrayHasKey('cta translate node page', $permissions,
      'Page permissions should exist initially.');

    // Disable content translation for page.
    $this->container->get('content_translation.manager')
      ->setEnabled('node', 'page', FALSE);

    // Get permissions again.
    $permissions_generator = Permissions::create($this->container);
    $permissions = $permissions_generator->ctaPermissions();

    // Verify page permissions are removed.
    $this->assertArrayNotHasKey('cta translate node page', $permissions,
      'Page permissions should be removed after disabling translation.');
    $this->assertArrayNotHasKey('cta create translation node page', $permissions,
      'Page permissions should be removed after disabling translation.');
    $this->assertArrayNotHasKey('cta delete translation node page', $permissions,
      'Page permissions should be removed after disabling translation.');
  }

}
