<?php

namespace Drupal\content_view_access\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to handle view access control.
 */
class ContentViewAccessSubscriber implements EventSubscriberInterface, ContainerInjectionInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new ContentViewAccessSubscriber.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountProxyInterface $current_user) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onRequest', 32],
    ];
  }

  /**
   * Handles the kernel request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event to process.
   */
  public function onRequest(RequestEvent $event) {
    $request = $event->getRequest();
    $route_name = $request->attributes->get('_route');

    if (in_array($route_name, ['entity.node.canonical', 'entity.taxonomy_term.canonical'])) {
      $entity = $request->attributes->get('node') ?? $request->attributes->get('taxonomy_term');

      if ($entity instanceof EntityInterface) {
        $entity_type = $entity->getEntityTypeId();
        $bundle = $entity->bundle();
        $user_roles = $this->currentUser->getRoles();

        $config = $this->configFactory->get('content_view_access.settings');
        $access = $config->get('access') ?? [];

        foreach ($user_roles as $role) {
          if (isset($access[$entity_type][$bundle][$role])) {
            $action = $access[$entity_type][$bundle][$role];

            switch ($action) {
              case '403':
                throw new AccessDeniedHttpException();

              case '404':
                throw new NotFoundHttpException();

              case 'front':
                $event->setResponse(new RedirectResponse('/'));
                $event->stopPropagation();
                return;

              case 'blank':
                $event->setResponse(new Response(''));
                $event->stopPropagation();
                return;
            }
          }
        }
      }
    }
  }

}
