<?php

namespace Drupal\content_view_access\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Configure content view access settings for this site.
 */
class ContentViewAccessForm extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a ContentViewAccessForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'content_view_access_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $vocabularies = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();
    $actions = [
      '' => '- None -',
      '403' => 'Access Denied (403)',
      '404' => 'Not Found (404)',
      'front' => 'Front Page',
      'blank' => 'Blank Page',
    ];
    $config = $this->configFactory->getEditable('content_view_access.settings');
    $access = $config->get('access') ?? [];

    $form['description'] = ['#markup' => '<p>Set view access rules for each bundle and user role. Leave empty for default behavior.</p>'];

    foreach (['node' => $content_types, 'taxonomy_term' => $vocabularies] as $entity_type => $bundles) {
      foreach ($bundles as $bundle_id => $bundle) {
        $has_values = !empty($access[$entity_type][$bundle_id]);
        $form[$entity_type . '__' . $bundle_id] = [
          '#type' => 'details',
          '#title' => $bundle->label() . " ($entity_type)",
          '#open' => $has_values,
        ];

        foreach ($roles as $role_id => $role) {
          $input_key = $entity_type . '__' . $bundle_id . '__' . $role_id;
          $form[$entity_type . '__' . $bundle_id][$input_key] = [
            '#type' => 'select',
            '#title' => $role->label(),
            '#options' => $actions,
            '#default_value' => $access[$entity_type][$bundle_id][$role_id] ?? '',
          ];
        }
      }
    }

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $vocabularies = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();

    $access = [];

    foreach (['node' => $content_types, 'taxonomy_term' => $vocabularies] as $entity_type => $bundles) {
      foreach ($bundles as $bundle_id => $bundle) {
        foreach ($roles as $role_id => $role) {
          $input_key = $entity_type . '__' . $bundle_id . '__' . $role_id;
          $action = $form_state->getValue($input_key);
          if (!empty($action)) {
            $access[$entity_type][$bundle_id][$role_id] = $action;
          }
        }
      }
    }

    $this->configFactory->getEditable('content_view_access.settings')->set('access', $access)->save();
    $this->messenger()->addStatus($this->t('Content view access configuration has been saved.'));
  }

}
