<?php

namespace Drupal\Tests\content_view_access\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\user\RoleInterface;

/**
 * Tests the Content View Access configuration and access control.
 *
 * @group content_view_access
 */
class ContentViewAccessTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'taxonomy', 'user', 'content_view_access'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests access control configuration for anonymous users.
   */
  public function testAnonymousUserAccess() {
    // Create an article.
    $node = $this->drupalCreateNode(['type' => 'article']);

    // Set config to restrict anonymous access.
    $this->config('content_view_access.settings')
      ->set('access', [
        'node' => [
          'article' => [
            'anonymous' => '403',
          ],
        ],
      ])
      ->save();

    // Log out to simulate anonymous user.
    $this->drupalLogout();

    // Attempt to access the node page.
    $this->drupalGet($node->toUrl());
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests access control for authenticated users.
   */
  public function testAuthenticatedUserAccess() {
    // Create a new role.
    $test_role = $this->drupalCreateRole([], 'test_role', 'test_role');

    // Create a user with the new role.
    $user = $this->drupalCreateUser([]);
    $user->addRole($test_role);
    $user->save();
    $this->drupalLogin($user);

    // Create an article.
    $node = $this->drupalCreateNode(['type' => 'article']);

    // With no specific rule, access should be granted.
    $this->drupalGet($node->toUrl());
    $this->assertSession()->statusCodeEquals(200);

    // Set config to restrict access for the new role.
    $this->config('content_view_access.settings')
      ->set('access', [
        'node' => [
          'article' => [
            $test_role => '403',
          ],
        ],
      ])
      ->save();

    // Attempt to access the node page.
    $this->drupalGet($node->toUrl());
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests access control for taxonomy terms.
   */
  public function testTaxonomyTermAccess() {
    // Create a vocabulary.
    $vocabulary = $this->createVocabulary();

    // Create a term.
    $term = $this->createTerm($vocabulary);

    // Set a redirect rule for anonymous users.
    $this->config('content_view_access.settings')
      ->set('access', [
        'taxonomy_term' => [
          $vocabulary->id() => [
            RoleInterface::ANONYMOUS_ID => '404',
          ],
        ],
      ])
      ->save();

    // Log out to simulate anonymous user.
    $this->drupalLogout();

    // Attempt to access the term page.
    $this->drupalGet($term->toUrl());
    $this->assertSession()->statusCodeEquals(404);
  }

  /**
   * Tests the different access denied actions.
   */
  public function testDifferentActions() {
    // Create an article.
    $node = $this->drupalCreateNode(['type' => 'article']);

    $actions = ['403', '404', 'front', 'blank'];

    foreach ($actions as $action) {
      // Set the access rule for anonymous users.
      $this->config('content_view_access.settings')
        ->set('access', [
          'node' => [
            'article' => [
              RoleInterface::ANONYMOUS_ID => $action,
            ],
          ],
        ])
        ->save();

      // Log out to simulate anonymous user.
      $this->drupalLogout();

      // Attempt to access the node page.
      $this->drupalGet($node->toUrl());

      if ($action === 'blank') {
        $this->assertSession()->pageTextEquals('');
        $this->assertSession()->statusCodeEquals(200);
      }
      else {
        $this->assertSession()->statusCodeEquals($action);
      }
    }
  }

  /**
   * Creates a vocabulary for testing.
   */
  protected function createVocabulary() {
    $vocabulary_storage = \Drupal::entityTypeManager()->getStorage('taxonomy_vocabulary');
    $vocabulary = $vocabulary_storage->create([
      'vid' => 'test_vocabulary',
      'name' => 'Test Vocabulary',
    ]);
    $vocabulary->save();
    return $vocabulary;
  }

  /**
   * Creates a term for testing.
   */
  protected function createTerm($vocabulary) {
    $term_storage = \Drupal::entityTypeManager()->getStorage('taxonomy_term');
    $term = $term_storage->create([
      'vid' => $vocabulary->id(),
      'name' => 'Test Term',
    ]);
    $term->save();
    return $term;
  }

}
