<?php

namespace Drupal\context_breadcrumb\ContextProvider;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Plugin\Context\Context;
use Drupal\Core\Plugin\Context\ContextProviderInterface;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\taxonomy\Entity\Term;

/**
 * Sets the current taxonomy term as a context on taxonomy term routes.
 */
class VocabularyContext implements ContextProviderInterface {

  use StringTranslationTrait;

  /**
   * The route match object.
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Constructs a new TaxonomyTermRouteContext.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match object.
   */
  public function __construct(RouteMatchInterface $route_match) {
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public function getRuntimeContexts(array $unqualified_context_ids): array {
    $result = [];
    $context_definition = new EntityContextDefinition('entity:taxonomy_term', NULL, FALSE);
    $value = NULL;
    if (($route_object = $this->routeMatch->getRouteObject()) && ($route_contexts = $route_object->getOption('parameters')) && isset($route_contexts['taxonomy_term'])) {
      if ($taxonomy_term = $this->routeMatch->getParameter('taxonomy_term')) {
        $value = $taxonomy_term;
      }
    }
    elseif ($this->routeMatch->getRouteName() == 'entity.taxonomy_term.add_form') {
      $vocabulary = $this->routeMatch->getParameter('taxonomy_vocabulary');
      $value = Term::create(['vid' => $vocabulary->id()]);
    }

    $cacheability = new CacheableMetadata();
    $cacheability->setCacheContexts(['route', 'url.site']);

    $context = new Context($context_definition, $value);
    $context->addCacheableDependency($cacheability);
    $result['taxonomy_term'] = $context;

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableContexts(): array {
    $context = new Context(new EntityContextDefinition('entity:taxonomy_term', $this->t('Taxonomy Term from URL')));
    return ['taxonomy_term' => $context];
  }

}
