<?php

namespace Drupal\context_breadcrumb\Plugin\ContextReaction;

use Drupal\context\ContextReactionPluginBase;
use Drupal\context_breadcrumb\Breadcrumb\ContextBreadcrumbBuilder;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Renderer;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a content reaction that adds breadcrumb to page.
 *
 * @ContextReaction(
 *   id = "context_breadcrumb",
 *   label = @Translation("Breadcrumb")
 * )
 */
class Breadcrumb extends ContextReactionPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The renderer.
   */
  protected Renderer $renderer;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ModuleHandlerInterface $module_handler,
    Renderer $renderer,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleHandler = $module_handler;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $permission = [
      'breadcrumbs' => [],
      'cache_query_args' => '',
    ];

    return parent::defaultConfiguration() + $permission;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): TranslatableMarkup {
    return $this->t('Context breadcrumb');
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    return $this->getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $breadcrumbs = $this->getConfiguration()['breadcrumbs'];
    $form['breadcrumbs'] = [
      '#type' => 'table',
      '#header' => [
        '',
        $this->t('Title'),
        $this->t('Url'),
        $this->t('Token'),
        $this->t('Weight'),
      ],
      '#rows' => [],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'table-sort-weight',
        ],
      ],
    ];

    for ($i = 0; $i <= 8; $i++) {
      // Mark the table row as draggable.
      $form['breadcrumbs'][$i]['#attributes']['class'][] = 'draggable';
      // Sort the table row according to its existing/configured weight.
      $form['breadcrumbs'][$i]['#weight'] = $breadcrumbs[$i]['weight'] ?? $i;

      $form['breadcrumbs'][$i]['drag'] = [
        '#markup' => ' ',
      ];

      $form['breadcrumbs'][$i]['title'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Title'),
        '#rows' => 1,
        '#title_display' => 'invisible',
        '#default_value' => $breadcrumbs[$i]['title'] ?? '',
      ];
      $form['breadcrumbs'][$i]['url'] = [
        '#type' => 'textarea',
        '#title' => $this->t('URL'),
        '#rows' => 1,
        '#title_display' => 'invisible',
        '#default_value' => $breadcrumbs[$i]['url'] ?? '',
      ];

      $form['breadcrumbs'][$i]['token'] = [
        '#type' => 'select',
        '#required' => FALSE,
        '#title' => $this->t('Token'),
        '#title_display' => 'invisible',
        '#default_value' => !empty($breadcrumbs[$i]['token']) ? 1 : '',
        '#empty_value' => '',
        '#options' => [
          '' => $this->t('None'),
          1 => $this->t('Yes'),
        ],
      ];

      $form['breadcrumbs'][$i]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for this row'),
        '#title_display' => 'invisible',
        '#default_value' => !empty($breadcrumbs[$i]['weight']) ? $breadcrumbs[$i]['weight'] : $i,
        // Classify the weight element for #tabledrag.
        '#attributes' => ['class' => ['table-sort-weight']],
      ];
    }

    if ($this->moduleHandler->moduleExists("token")) {
      $token_tree = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['node', 'user', 'term', 'vocabulary'],
      ];

      $rendered_token_tree = $this->renderer->render($token_tree);
      $form['description']['#type'] = 'item';
      $form['description']['#description'] = t('This field supports tokens. @browse_tokens_link', [
        '@browse_tokens_link' => $rendered_token_tree,
      ]);
    }

    $form['cache_query_args'] = [
      '#type' => 'textarea',
      '#required' => FALSE,
      '#title' => $this->t('Cache query args'),
      '#description' => $this->t('Query args will be applied to the breadcrumb, each argument per line. Example id, uid. Use !all for cache everything.',),
      '#cols' => 60,
      '#rows' => 3,
      '#default_value' => $this->getConfiguration()['cache_query_args'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    // Validate for breadcrumbs.
    foreach ($form_state->getValue('breadcrumbs') as $i => $breadcrumb) {
      if (mb_strlen($breadcrumb['title']) && !mb_strlen($breadcrumb['url'])) {
        $form_state->setErrorByName('breadcrumbs][' . $i . '][url', $this->t('@name field is required.', ['@name' => 'Url']));
      }
      if (!mb_strlen($breadcrumb['title']) && mb_strlen($breadcrumb['url'])) {
        $form_state->setErrorByName('breadcrumbs][' . $i . '][token', $this->t('@name field is required.', ['@name' => 'Token']));
      }

      if (mb_strlen($breadcrumb['title']) && mb_strlen($breadcrumb['url'])) {
        if (ContextBreadcrumbBuilder::isToken($breadcrumb['url']) && empty($breadcrumb['token'])) {
          $form_state->setErrorByName('breadcrumbs][' . $i . '][token', $this->t('The url using token, please select token option.'));
        }
        if (ContextBreadcrumbBuilder::isToken($breadcrumb['title']) && empty($breadcrumb['token'])) {
          $form_state->setErrorByName('breadcrumbs][' . $i . '][token', $this->t('The title using token, please select token option.'));
        }
        if (!ContextBreadcrumbBuilder::isToken($breadcrumb['url'])
          && !in_array($breadcrumb['url'], ['<front>', '<nolink>'])
          && !str_contains($breadcrumb['url'], 'http://')
          && !str_contains($breadcrumb['url'], 'https://')
          && $breadcrumb['url'][0] !== '/') {
          $form_state->setErrorByName('breadcrumbs][' . $i . '][url', $this->t('The url path has to start with a slash.'));
        }
      }
    }

    // Validate for cache_query_args.
    $value = trim($form_state->getValue('cache_query_args'));
    if ($value === '') {
      return;
    }

    $lines = preg_split('/\r\n|\r|\n/', $value);
    foreach ($lines as $line_num => $arg) {
      $arg = trim($arg);
      if ($arg === '') {
        continue;
      }

      // Special keyword !all is allowed.
      if ($arg === '!all') {
        continue;
      }

      // Only allow word characters (letters, digits, underscore).
      if (!preg_match('/^[A-Za-z0-9_]+$/', $arg)) {
        $form_state->setErrorByName(
          'cache_query_args',
          $this->t('Invalid query argument "@arg" on line @line. Only alphanumeric and underscore are allowed, or use !all.', [
            '@arg' => $arg,
            '@line' => $line_num + 1,
          ])
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $breadcrumbs = $form_state->getValue('breadcrumbs');

    // Re-order the crumbs based on the weight's value.
    usort($breadcrumbs, function ($a, $b) {
      if ($a['weight'] == $b['weight']) {
        return 0;
      }
      return $a['weight'] < $b['weight'] ? -1 : 1;
    });

    // Update the weight values to be consistent to the index.
    for ($i = 0; $i <= 8; $i++) {
      $breadcrumbs[$i]['weight'] = $i;
    }

    // Update the indexes.
    $breadcrumbs = array_values($breadcrumbs);

    $form_state->setValue('breadcrumbs', $breadcrumbs);

    $this->setConfiguration([
      'breadcrumbs' => $form_state->getValue('breadcrumbs'),
      'cache_query_args' => trim($form_state->getValue('cache_query_args') ?? ''),
    ]);
    Cache::invalidateTags(['context:breadcrumb']);
  }

}
