<?php

namespace Drupal\context_menu_parent_trail\Plugin\Condition;

use Drupal\Core\Condition\ConditionPluginBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuActiveTrailInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Context condition: "Menu parent in active trail".
 *
 * @Condition(
 *   id = "menu_parent_active_trail",
 *   label = @Translation("Menu parent in active trail"),
 *   context_definitions = {}
 * )
 */
class MenuParentActive extends ConditionPluginBase implements ContainerFactoryPluginInterface
{

  /**
   * @var \Drupal\Core\Menu\MenuActiveTrailInterface
   */
  protected $activeTrail;

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    MenuActiveTrailInterface $active_trail,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->activeTrail = $active_trail;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('menu.active_trail'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration()
  {
    return [
      'menu_name' => 'main',
      'parent_uuid' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state)
  {
    // Available menus.
    $menus = [];
    $menu_entities = $this->entityTypeManager->getStorage('menu')->loadMultiple();
    foreach ($menu_entities as $machine_name => $menu) {
      $menus[$machine_name] = $menu->label() . " ($machine_name)";
    }

    $form['menu_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu'),
      '#options' => $menus,
      '#default_value' => $this->configuration['menu_name'] ?? 'main',
      '#required' => TRUE,
    ];

    // Autocomplete for parent item (menu_link_content).
    $form['parent_link'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Parent menu item'),
      '#target_type' => 'menu_link_content',
      '#default_value' => $this->getParentEntity(),
      '#description' => $this->t('Choose the parent menu link. The condition is TRUE if this link appears in the active trail (all its descendants match).'),
      '#required' => TRUE,
      // Filter by menu using the correct selection_settings.
      '#selection_settings' => [
        'menu_name' => $this->configuration['menu_name'] ?? 'main',
      ],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void
  {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['menu_name'] = $form_state->getValue('menu_name');

    $parent_value = $form_state->getValue('parent_link');
    $storage = $this->entityTypeManager->getStorage('menu_link_content');
    $parent = NULL;

    if ($parent_value instanceof \Drupal\menu_link_content\MenuLinkContentInterface) {
      $parent = $parent_value;
    } elseif (is_array($parent_value) && !empty($parent_value['target_id'])) {
      $parent = $storage->load($parent_value['target_id']);
    } elseif (!empty($parent_value)) {
      $parent = $storage->load($parent_value);
    }

    $this->configuration['parent_uuid'] = $parent ? $parent->uuid() : '';
  }


  /**
   * Load the entity selected by UUID to preload autocomplete.
   */
  protected function getParentEntity(): ?MenuLinkContent
  {
    $uuid = $this->configuration['parent_uuid'] ?? '';
    if (!$uuid) {
      return NULL;
    }

    $storage = $this->entityTypeManager->getStorage('menu_link_content');
    $ids = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('uuid', $uuid)
      ->range(0, 1)
      ->execute();

    if ($ids) {
      return $storage->load(reset($ids));
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate()
  {
    if (empty($this->configuration['menu_name']) || empty($this->configuration['parent_uuid'])) {
      // Respect negation when there is no configuration.
      return $this->isNegated() ? TRUE : FALSE;
    }

    $menu_name = $this->configuration['menu_name'];
    $needle = 'menu_link_content:' . $this->configuration['parent_uuid'];

    // Returns the array of plugin IDs in the active trail of the menu.
    $trail = $this->activeTrail->getActiveTrailIds($menu_name);
    $present = in_array($needle, $trail, TRUE);

    return $this->isNegated() ? !$present : $present;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts()
  {
    // It varies depending on the active trail of the current route and the selected menu.
    $contexts = parent::getCacheContexts();
    $contexts[] = 'route.menu_active_trails';
    $contexts[] = 'url.path';
    return $contexts;
  }

  /**
   * {@inheritdoc}
   */
  public function summary()
  {
    $menu = $this->configuration['menu_name'] ?? 'main';
    return $this->t('Menu parent in active trail (@menu)', ['@menu' => $menu]);
  }
}
