<?php

declare(strict_types=1);

namespace Drupal\context_profile_role\Plugin\Condition;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Condition\Attribute\Condition;
use Drupal\Core\Condition\ConditionPluginBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides a 'User Profile Role' condition.
 */
#[Condition(
  id: 'user_profile_role',
  label: new TranslatableMarkup('User Profile Role'),
  context_definitions: [
    'user_profile' => new EntityContextDefinition(
      data_type: 'entity:user',
      label: new TranslatableMarkup('User Profile Role'),
    ),
  ]
)]
class UserProfileRole extends ConditionPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\user\RoleStorageInterface $roleStorage */
    $roleStorage = $this->entityTypeManager->getStorage('user_role');

    $options = [];
    $user_roles = $roleStorage->loadMultiple();
    foreach ($user_roles as $role) {
      $options[$role->id()] = $role->label();
    }
    $form['roles'] = [
      '#title' => $this->t('Roles'),
      '#type' => 'checkboxes',
      '#options' => $options,
      '#default_value' => $this->configuration['roles'],
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    /** @var array $form_state_roles_value */
    $form_state_roles_value = $form_state->getValue('roles');
    $this->configuration['roles'] = \array_filter($form_state_roles_value);
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): MarkupInterface {
    /** @var string[] $roles */
    $roles = $this->configuration['roles'];
    if (\count($roles) > 1) {
      $last = \array_pop($roles);
      $roles = \implode(', ', $roles);
      return $this->t(
        'The user role is @roles or @last',
        ['@roles' => $roles, '@last' => $last]
      );
    }
    $bundle = \reset($roles);
    return $this->t('The user role is @bundle', ['@bundle' => $bundle]);
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    /** @var string[] $roles */
    $roles = $this->configuration['roles'];
    if (empty($roles) && !$this->isNegated()) {
      return TRUE;
    }
    $user_profile_has_role = FALSE;
    /** @var \Drupal\user\UserInterface|null $user_profile */
    $user_profile = $this->getContextValue('user_profile');
    if ($user_profile) {
      $user_profile_roles = $user_profile->getRoles();
      // Search if the user profile has one of the authorized roles.
      foreach ($roles as $expected_role) {
        if (\in_array($expected_role, $user_profile_roles, TRUE)) {
          $user_profile_has_role = TRUE;
          break;
        }
      }
    }

    return $user_profile_has_role;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return ['roles' => []] + parent::defaultConfiguration();
  }

}
