<?php

namespace Drupal\conword\Form;

use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Executable\ExecutableManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The setting form for the conword settings.
 *
 * @package Drupal\conword\Form
 */
class ConwordConfigForm extends ConfigFormBase {

  const CONFIG_OBJECT_NAME = 'conword.settings';

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Executable\ExecutableManagerInterface
   */
  protected ExecutableManagerInterface $conditionPluginManager;

  /**
   * The visibility collection.
   *
   * @var \Drupal\Core\Condition\ConditionPluginCollection
   */
  protected ConditionPluginCollection $visibilityCollection;

  /**
   * The context manager service.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected ContextRepositoryInterface $contextRepository;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->conditionPluginManager = $container->get('plugin.manager.condition');
    $instance->contextRepository = $container->get('context.repository');
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * Gets conditions for this block.
   *
   * @return \Drupal\Core\Condition\ConditionPluginCollection
   *   An array or collection of configured condition plugins.
   */
  public function getVisibilityConditions(): ConditionPluginCollection {
    if (!isset($this->visibilityCollection)) {
      $this->visibilityCollection = new ConditionPluginCollection($this->conditionPluginManager, $this->config(self::CONFIG_OBJECT_NAME)->get('visibility') ?? []);
    }
    return $this->visibilityCollection;
  }

  /**
   * Returns an array of visibility condition configurations.
   *
   * @return array
   *   An array of visibility condition configuration keyed by the condition ID.
   */
  public function getVisibility(): array {
    return $this->getVisibilityConditions()->getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'conword_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      self::CONFIG_OBJECT_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Store the gathered contexts in the form state for other objects to use
    // during form building.
    $form_state->setTemporaryValue('gathered_contexts', $this->contextRepository->getAvailableContexts());

    $form['#tree'] = TRUE;
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General settings'),
      '#open' => TRUE,
    ];

    $form['general']['customerId'] = [
      '#title' => $this->t('Customer ID'),
      '#description' => $this->t('Enter your conword customer ID.'),
      '#type' => 'textfield',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':customer_id',
      '#required' => TRUE,
    ];

    $form['general']['disableLanguageSwitcher'] = [
      '#title' => $this->t('Disable language switcher'),
      '#description' => $this->t('Disables default language switcher'),
      '#type' => 'checkbox',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':conwordConfig.disable_language_switcher',
    ];

    $form['general']['disableRtlAttribute'] = [
      '#title' => $this->t('Disable RTL attribute'),
      '#description' => $this->t('Disables RTL attribute'),
      '#type' => 'checkbox',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':conwordConfig.disable_rtl_attribute',
    ];

    $form['notice'] = [
      '#type' => 'container',
      'children' => [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [
            $this->t('<em>Visibility</em> settings is bypassed by the block plugin if <em>Disable language switcher</em> ist activated.'),
          ],
        ],
      ],
      '#states' => [
        'visible' => [
          ':input[name="general[disableLanguageSwitcher]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Visibility settings.
    $form['visibility'] = $this->buildVisibilityInterface([], $form_state);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Helper function for building the visibility UI form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form array with the visibility UI added in.
   */
  protected function buildVisibilityInterface(array $form, FormStateInterface $form_state): array {
    $form['visibility_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Visibility'),
      '#parents' => ['visibility_tabs'],
      '#attached' => [
        'library' => [
          'conword/conword.admin',
        ],
      ],
    ];

    $visibility = $this->getVisibility();
    $conditionDefinitions = $this->conditionPluginManager->getDefinitionsForContexts($form_state->getTemporaryValue('gathered_contexts'));
    foreach ($conditionDefinitions as $condition_id => $definition) {
      // Don't display the current theme condition.
      if ($condition_id == 'current_theme') {
        continue;
      }
      // Don't display the language condition until we have multiple languages.
      if ($condition_id == 'language' && !$this->languageManager->isMultilingual()) {
        continue;
      }

      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $this->conditionPluginManager->createInstance($condition_id, $visibility[$condition_id] ?? []);
      $form_state->set(['conditions', $condition_id], $condition);
      $condition_form = $condition->buildConfigurationForm([], $form_state);
      $condition_form['#type'] = 'details';
      $condition_form['#title'] = $condition->getPluginDefinition()['label'];
      $condition_form['#group'] = 'visibility_tabs';
      $form[$condition_id] = $condition_form;
    }

    // Disable negation for specific conditions.
    $disable_negation = [
      'entity_bundle:node',
      'language',
      'response_status',
      'user_role',
    ];

    foreach ($disable_negation as $condition) {
      if (isset($form[$condition])) {
        $form[$condition]['negate']['#type'] = 'value';
        $form[$condition]['negate']['#value'] = $form[$condition]['negate']['#default_value'];
      }
    }

    if (isset($form['user_role'])) {
      $form['user_role']['#title'] = $this->t('Roles');
      unset($form['user_role']['roles']['#description']);
    }

    if (isset($form['request_path'])) {
      $form['request_path']['#title'] = $this->t('Pages');
      $form['request_path']['negate']['#type'] = 'radios';
      $form['request_path']['negate']['#default_value'] = (int) $form['request_path']['negate']['#default_value'];
      $form['request_path']['negate']['#title_display'] = 'invisible';
      $form['request_path']['negate']['#options'] = [
        $this->t('Show for the listed pages'),
        $this->t('Hide for the listed pages'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    foreach ($form_state->getValue('visibility') as $condition_id => $values) {
      // Allow the condition to validate the form.
      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $form_state->get(['conditions', $condition_id]);
      $condition->validateConfigurationForm($form['visibility'][$condition_id], SubformState::createForSubform($form['visibility'][$condition_id], $form, $form_state));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    foreach ($form_state->getValue('visibility') as $condition_id => $values) {
      // Allow the condition to submit the form.
      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $form_state->get(['conditions', $condition_id]);
      $condition->submitConfigurationForm($form['visibility'][$condition_id], SubformState::createForSubform($form['visibility'][$condition_id], $form, $form_state));

      $condition_configuration = $condition->getConfiguration();
      // Update the visibility conditions on the block.
      $this->visibilityCollection->addInstanceId($condition_id, $condition_configuration);
    }

    // Set visibility settings explicitly due to no presence of
    // #config_target concept for plugin sub forms.
    $config = $this->config(self::CONFIG_OBJECT_NAME);
    $config->set('visibility', $this->visibilityCollection->getConfiguration());
    $config->save();
  }

}
