<?php

namespace Drupal\cookies_addons_fields\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Returns responses for Cookies Addons fields routes.
 */
class CookiesAddonsFieldsController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $displayRepo;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a CookiesAddonsFieldsController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repo
   *   The entity display repository.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $display_repo,
    RendererInterface $renderer,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->displayRepo = $display_repo;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('renderer')
    );
  }

  /**
   * Retrieves a specific field for an entity and returns an AJAX response.
   *
   * @param string $field_id
   *   The field ID in the format "entity_type-entity_id-field_name".
   * @param string $service
   *   The service name used for the field placeholder selector.
   * @param string $view_mode
   *   The view mode in which the field should be rendered.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   An AJAX response containing the rendered field replacement command.
   *
   * @throws \Exception
   *   Thrown when the field ID format is invalid, the entity cannot be loaded,
   *   or the specified field does not exist on the entity.
   */
  public function getField($field_id, $service, $view_mode) {
    // Parse the field ID to get entity type, entity ID, and field name.
    if (!preg_match('/^([a-z_]+)-([0-9]+)-([a-z0-9_]+)$/', $field_id, $m)) {
      throw new NotFoundHttpException();
    }
    // Extract entity type, entity ID, and field name.
    $entity_type = $m[1];
    $entity_id   = (int) $m[2];
    $field_name  = $m[3];

    // Load the entity.
    $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    if (!$entity instanceof ContentEntityInterface) {
      throw new NotFoundHttpException();
    }

    $account = $this->currentUser();
    $entity_access = $entity->access('view', $account, TRUE);
    if (!$entity_access->isAllowed()) {
      throw new AccessDeniedHttpException();
    }

    // Check if the entity has the field.
    if (!$entity->hasField($field_name)) {
      throw new NotFoundHttpException();
    }

    // Check if the field is accessible for the current user.
    $field = $entity->get($field_name);
    $field_access = $field->access('view', $account, TRUE);
    if (!$field_access->isAllowed()) {
      throw new AccessDeniedHttpException();
    }

    // Validate view mode against allowed options.
    $allowed_view_modes = array_keys($this->displayRepo->getViewModeOptions($entity_type));
    if (!in_array($view_mode, $allowed_view_modes, TRUE)) {
      throw new NotFoundHttpException();
    }

    // Build the field render array.
    $view_builder = $this->entityTypeManager->getViewBuilder($entity_type);
    $field_render_array = $view_builder->viewField($field, $view_mode);

    // Add cacheability so access decisions don’t leak.
    $this->renderer->addCacheableDependency($field_render_array, $entity_access);
    $this->renderer->addCacheableDependency($field_render_array, $field_access);
    $this->renderer->addCacheableDependency($field_render_array, $entity);

    // Create the selector for the field placeholder.
    $selector = sprintf(
      '.cookies-addons-fields-placeholder[data-cookies-service="%s"][data-field-id="%s"]',
      $service,
      $field_id
    );

    $response = new AjaxResponse();
    // Add the replace command to the AJAX response.
    $response->addCommand(new ReplaceCommand($selector, $field_render_array));

    return $response;
  }

}
