<?php

declare(strict_types=1);

namespace Drupal\copyright_notice\Hook;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Defines hooks for the Copyright Notice module.
 */
class CopyrightNoticeHooks {

  use StringTranslationTrait;

  /**
   * Constructs a new CopyrightNoticeHooks object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name, RouteMatchInterface $route_match): ?string {
    if ($route_name === 'help.page.copyright_notice') {
      $output = '<h2>' . $this->t('About') . '</h2>';
      $output .= '<p>' . $this->t('The Copyright Notice Field module provides a simple way to display a copyright notice on your Drupal site. You can customize the name, year, and whether to display the current year.  See the <a href=":field">Field module help</a> and the <a href=":field_ui">Field UI help</a> pages for general information on fields and how to create and manage them. For more information, see the <a href=":project_link">online documentation for the Copyright Notice Field module</a>.', [
        ':field' => Url::fromRoute('help.page', [
          'name' => 'field',
        ])->toString(),
        ':field_ui' => $this->moduleHandler->moduleExists('field_ui') ? Url::fromRoute('help.page', [
          'name' => 'field_ui',
        ])->toString() : '#',
        ':project_link' => 'https://www.drupal.org/project/copyright_notice',
      ]) . '</p>';
      $output .= '<h2>' . $this->t('Uses') . '</h2>';
      $output .= '<dl>';
      $output .= '<dt>' . $this->t('Managing and displaying copyright notice fields') . '</dt>';
      $output .= '<dd>' . $this->t('The <em>settings</em> and the <em>display</em> of the copyright notice field can be configured separately. See the <a href=":field_ui">Field UI help</a> for more information on how to manage fields and their display.', [
        ':field_ui' => $this->moduleHandler->moduleExists('field_ui') ? Url::fromRoute('help.page', [
          'name' => 'field_ui',
        ])->toString() : '#',
      ]) . '</dd>';
      $output .= '<dt>' . $this->t('Displaying the current year') . '</dt>';
      $output .= '<dd>' . $this->t('Enable the current year in the copyright notice using the checkbox on the <em>Manage display</em> page.') . '</dd>';
      $output .= '<dt>' . $this->t('Displaying the site name') . '</dt>';
      $output .= '<dd>' . $this->t('Include the site name in the copyright notice using the checkbox on the <em>Manage display</em> page.') . '</dd>';
      $output .= '</dl>';
      return $output;
    }
    return NULL;
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'copyright_notice' => [
        'variables' => [
          'name' => NULL,
          'year' => NULL,
          'display_current_year' => FALSE,
          'attributes' => [],
        ],
        'initial preprocess' => static::class . ':preprocessCopyrightNotice',
      ],
    ];
  }

  /**
   * Prepares variables for copyright notice templates.
   *
   * Default template: copyright-notice.html.twig.
   *
   * @parm array $variables
   *   An associative array containing:
   *   - name: the name to be printed in the copyright notice, usually this is
   *     the site's name.
   *   - year: the year to be displayed, typically the founding year of the
   *     company/website.
   *   - display_current_year: boolean indicating whether the current year
   *     (interval) should be displayed.
   *   - attributes: associative array of additional attributes for the
   *     surrounding HTML tag.
   */
  public function preprocessCopyrightNotice(array &$variables): void {
    $start_year = (int) ($variables['year'] ?? 0);

    // When the current year should also be displayed, make adjustments to the
    // $variables['year'] variable and ensure correct cache invalidation happens
    // at least at the end of the year.
    if ($variables['display_current_year']) {
      // Get the current date and time as an object.
      $now = new DrupalDateTime();

      // Get the current year.
      $current_year = (int) $now->format('Y');

      // Set the year property dependent on the values set for the start and
      // current year.
      if (!empty($start_year)) {
        if ($start_year < $current_year) {
          $variables['year'] = $start_year . ' - ' . $current_year;
        }
        elseif ($start_year > $current_year) {
          $variables['year'] = $current_year . ' - ' . $start_year;
        }
        else {
          $variables['year'] = $current_year;
        }
      }
      else {
        $variables['year'] = $current_year;
      }

      // Get a time representation object for next year.
      $next_year = new DrupalDateTime('1 jan next year');

      // Compute the number of seconds left this year, before the year changes.
      $interval = $next_year->diff($now);
      $seconds_left = $interval->days * 86400 + $interval->h * 3600 + $interval->i * 60 + $interval->s * $interval->invert;

      // The cache value should be cleared, once a new year starts.
      $variables['#cache']['max-age'] = $seconds_left;
    }
  }

}
