<?php

declare(strict_types=1);

namespace Drupal\Tests\copyright_notice\Kernel;

use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests rendering of the #copyright_notice theme hook.
 */
#[Group('copyright_notice')]
class RenderTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['copyright_notice'];

  /**
   * Tests rendering of the #copyright_notice theme hook.
   */
  #[DataProvider('copyRightNoticeProvider')]
  public function testCopyrightNoticeRendering(array $element, string $expected): void {
    $this->assertThemeOutput('copyright_notice', $element, $expected);
  }

  /**
   * Provides data for testCopyrightNoticeRendering().
   *
   * @return \Generator
   *   An array of test data:
   *     - render element containing the #copyright_notice theme hook
   *     - expected HTML output
   */
  public static function copyRightNoticeProvider(): \Generator {
    $current_year = (int) date('Y');

    yield 'No properties are set' => [
      [
        '#theme' => 'copyright_notice',
      ],
      '<span>Copyright &copy;, all rights reserved.</span>',
    ];

    yield 'Only name is provided' => [
      [
        '#theme' => 'copyright_notice',
        '#name' => 'Test',
      ],
      '<span>Copyright &copy; Test, all rights reserved.</span>',
    ];

    yield 'Only base year is displayed' => [
      [
        '#theme' => 'copyright_notice',
        '#name' => 'Test',
        '#year' => 1982,
        '#display_current_year' => FALSE,
        '#attributes' => [],
      ],
      '<span>Copyright &copy; 1982 Test, all rights reserved.</span>',
    ];

    yield 'Only current year is displayed' => [
      [
        '#theme' => 'copyright_notice',
        '#name' => 'Test',
        '#year' => '',
        '#display_current_year' => TRUE,
        '#attributes' => [],
      ],
      '<span>Copyright &copy; ' . $current_year . ' Test, all rights reserved.</span>',
    ];

    yield 'Base and current year are displayed' => [
      [
        '#theme' => 'copyright_notice',
        '#name' => 'Test',
        '#year' => 1982,
        '#display_current_year' => TRUE,
        '#attributes' => [],
      ],
      '<span>Copyright &copy; 1982 - ' . $current_year . ' Test, all rights reserved.</span>',
    ];

    yield 'Base and current year are the same' => [
      [
        '#theme' => 'copyright_notice',
        '#name' => 'Test',
        '#year' => $current_year,
        '#display_current_year' => TRUE,
        '#attributes' => [],
      ],
      '<span>Copyright &copy; ' . $current_year . ' Test, all rights reserved.</span>',
    ];

    yield 'Base year is in the future' => [
      [
        '#theme' => 'copyright_notice',
        '#name' => 'Test',
        '#year' => $current_year + 1,
        '#display_current_year' => TRUE,
        '#attributes' => [],
      ],
      '<span>Copyright &copy; ' . $current_year . ' - ' . ($current_year + 1) . ' Test, all rights reserved.</span>',
    ];
  }

}
