/**
 * @file
 * Defines JavaScript behaviors for the countdown module.
 */

(function ($, window, Drupal, once) {
  "use strict";

  const breakpoint = 1024;
  const storageMobile = 'Drupal.countdown.sidebarExpanded.mobile';
  const storageDesktop = 'Drupal.countdown.sidebarExpanded.desktop';

  /**
   * Provide the summary information for the countdown settings vertical tabs.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the behavior for the countdown settings summaries.
   */
  Drupal.behaviors.countdownSettingsSummary = {
    attach() {
      // The drupalSetSummary method required for this behavior is not available
      // on the countdown settings page, so we need to make sure this
      // behavior is processed only if drupalSetSummary is defined.
      if (typeof $.fn.drupalSetSummary === 'undefined') {
        return;
      }

      $(once('countdown-themes', '[data-drupal-selector="edit-theme-visibility"]'),
      ).drupalSetSummary((context) => {
        const $themes = $(context).find(
          'select[name="themes[]"]',
        );
        if (!$themes.length || !$themes[0].value) {
          return Drupal.t('Not restricted');
        }

        return Drupal.t('Restricted to !theme', { '!theme': $themes.val() });
      });

      $(once('countdown-pages', '[data-drupal-selector="edit-page-visibility"]'),
      ).drupalSetSummary((context) => {
        const $pages = $(context).find(
          'textarea[name="pages"]',
        );
        if (!$pages.length || !$pages[0].value) {
          return Drupal.t('Not restricted');
        }

        return Drupal.t('Restricted to certain pages');
      });
    },
  };

  /**
   * Persist the library status details open/closed state.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the behavior for persisting details state.
   */
  Drupal.behaviors.countdownLibraryStatusPersist = {
    attach(context) {
      // Find the library status details element with our stable selector.
      once('countdown-library-status-persist', 'details.countdown-library-status', context).forEach((details) => {
        // Find the hidden input that stores the state.
        const hiddenInput = document.querySelector('[data-drupal-selector="edit-hide-status"]');

        if (!hiddenInput) {
          return;
        }

        // Initialize the hidden input value based on current details state.
        // Value is '1' when closed (hidden), '0' when open.
        hiddenInput.value = details.open ? '0' : '1';

        // Listen for the native toggle event on the details element.
        details.addEventListener('toggle', function() {
          // Update the hidden input when details state changes.
          hiddenInput.value = this.open ? '0' : '1';
        });
      });
    }
  };

  Drupal.behaviors.countdownSidebar = {
    attach: function attach(context) {
      once('countdownHelpBlock', '.region-help .block-help', context).forEach(block => {
        const blockHelp = context.querySelector('.block-help');
        const newParent = context.querySelector('.region-help__items__inner');

        if (newParent && newParent.querySelectorAll('.block-help').length === 0) {
          newParent.appendChild(blockHelp);
        }
      });

      once('countdownForm', '#block-gin-content form.countdown-admin-settings', context).forEach(form => {
        const sticky = context.querySelector('.countdown-sticky');
        const newParent = context.querySelector('.region-sticky__items__inner');

        if (newParent && newParent.querySelectorAll('.countdown-sticky').length === 0) {
          newParent.appendChild(sticky);

          // Attach form elements to the main form
          const actionButtons = newParent.querySelectorAll('button, input, select, textarea');

          if (actionButtons.length > 0) {
            actionButtons.forEach((el) => {
              el.setAttribute('form', form.getAttribute('id'));
              el.setAttribute('id', el.getAttribute('id') + '--countdown');
            });
          }
        }
      });

      Drupal.countdownSidebar.init(context);
    },
  };

  Drupal.countdownSidebar = {
    init: function (context) {
      once('countdownSidebarInit', '#countdown-sidebar', context).forEach(() => {

        // If a variable does not exist, create it,
        // default being to show sidebar.
        if (!localStorage.getItem(storageDesktop)) {
          localStorage.setItem(storageDesktop, 'true');
        }

        // Set mobile initial to false.
        if (window.innerWidth >= breakpoint) {
          if (localStorage.getItem(storageDesktop) === 'true') {
            this.showSidebar();
          }
          else {
            this.collapseSidebar();
          }
        }

        // Show navigation with shortcut:
        // OPTION + S (Mac) / ALT + S (Windows)
        document.addEventListener('keydown', e => {
          if (e.altKey === true && e.code === 'KeyS') {
            this.toggleSidebar();
          }
        });

        window.onresize = Drupal.debounce(this.handleResize, 150);
      });

      // Toolbar toggle
      once('countdownSidebarToggle', '.meta-sidebar__trigger', context).forEach(el => el.addEventListener('click', e => {
        e.preventDefault();
        this.removeInlineStyles();
        this.toggleSidebar();
      }));

      // Toolbar close
      once('countdownSidebarClose', '.meta-sidebar__close, .meta-sidebar__overlay', context).forEach(el => el.addEventListener('click', e => {
        e.preventDefault();
        this.removeInlineStyles();
        this.collapseSidebar();
      }));
    },

    toggleSidebar: () => {
      // Set active state.
      if (document.querySelector('.meta-sidebar__trigger').classList.contains('is-active')) {
        Drupal.countdownSidebar.collapseSidebar();
      }
      else {
        Drupal.countdownSidebar.showSidebar();
      }
    },

    showSidebar: () => {
      const chooseStorage = window.innerWidth < breakpoint ? storageMobile : storageDesktop;
      const showLabel = Drupal.t('Hide sidebar panel');
      const sidebarTrigger = document.querySelector('.meta-sidebar__trigger');

      sidebarTrigger.setAttribute('title', showLabel);
      sidebarTrigger.querySelector('span').innerHTML = showLabel;
      sidebarTrigger.setAttribute('aria-expanded', 'true');
      sidebarTrigger.classList.add('is-active');

      document.body.setAttribute('data-meta-sidebar', 'open');

      // Expose to localStorage.
      localStorage.setItem(chooseStorage, 'true');
    },

    collapseSidebar: () => {
      const chooseStorage = window.innerWidth < breakpoint ? storageMobile : storageDesktop;
      const hideLabel = Drupal.t('Show sidebar panel');
      const sidebarTrigger = document.querySelector('.meta-sidebar__trigger');

      sidebarTrigger.setAttribute('title', hideLabel);
      sidebarTrigger.querySelector('span').innerHTML = hideLabel;
      sidebarTrigger.setAttribute('aria-expanded', 'false');
      sidebarTrigger.classList.remove('is-active');

      document.body.setAttribute('data-meta-sidebar', 'closed');

      // Expose to localStorage.
      localStorage.setItem(chooseStorage, 'false');
    },

    handleResize: () => {
      Drupal.countdownSidebar.removeInlineStyles();

      // If small viewport, always collapse sidebar.
      if (window.innerWidth < breakpoint) {
        Drupal.countdownSidebar.collapseSidebar();
      } else {
        // If large viewport, show sidebar if it was open before.
        if (localStorage.getItem(storageDesktop) === 'true') {
          Drupal.countdownSidebar.showSidebar();
        } else {
          Drupal.countdownSidebar.collapseSidebar();
        }
      }
    },

    removeInlineStyles: () => {
      // Remove init styles.
      const elementToRemove = document.querySelector('.gin-sidebar-inline-styles');
      if (elementToRemove) {
        elementToRemove.parentNode.removeChild(elementToRemove);
      }
    },

  };
})(jQuery, window, Drupal, once);
