/**
 * @file
 * Core integration loader for the Countdown module.
 *
 * This file provides the main behavior and registry system for loading
 * library-specific integrations on demand. It implements lazy loading to
 * optimize performance by only loading the code needed for the active library.
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Registry and utility namespace for countdown functionality.
   *
   * @namespace
   */
  Drupal.countdown = Drupal.countdown || {
    loaders: {},
    loaded: {},
    instances: new WeakMap()
  };

  /**
   * Register a loader function for a specific library.
   *
   * @param {string} library
   *   The library identifier (e.g., 'tick', 'flipclock').
   * @param {Function} loader
   *   The loader function to initialize the library.
   */
  Drupal.countdown.registerLoader = function(library, loader) {
    this.loaders[library] = loader;
  };

  /**
   * Check if a library integration is already loaded.
   *
   * @param {string} library
   *   The library identifier.
   *
   * @return {boolean}
   *   TRUE if the integration is loaded, FALSE otherwise.
   */
  Drupal.countdown.isLoaded = function(library) {
    return !!this.loaded[library];
  };

  /**
   * Load a library integration dynamically.
   *
   * This function implements lazy loading by injecting the integration script
   * only when needed. It uses a callback pattern to handle async loading.
   *
   * @param {string} library
   *   The library identifier.
   * @param {Function} callback
   *   Callback to execute after the integration loads.
   */
  Drupal.countdown.loadIntegration = function(library, callback) {
    // Check if already loaded.
    if (this.loaded[library]) {
      callback();
      return;
    }

    // Check if loader is already registered (inline attachment).
    if (this.loaders[library]) {
      this.loaded[library] = true;
      callback();
      return;
    }

    // Build the integration script path.
    const basePath = drupalSettings.countdown.integrationBasePath || '/modules/contrib/countdown/js/integrations';
    const scriptPath = basePath + '/countdown.' + library + '.integration.js';

    // Create and inject the script element.
    const script = document.createElement('script');
    script.src = scriptPath;
    script.async = true;
    script.onload = () => {
      this.loaded[library] = true;
      callback();
    };
    script.onerror = () => {
      console.error('Countdown: Failed to load integration for', library);
    };

    document.head.appendChild(script);
  };

  /**
   * Initialize countdown timers for all matching elements.
   *
   * @param {Element} context
   *   The context element to search within.
   * @param {Object} settings
   *   The drupalSettings object.
   */
  Drupal.countdown.initialize = function(context, settings) {
    // Find all countdown timer elements.
    const elements = once('countdown-init', '.countdown-timer', context);

    elements.forEach(element => {
      // Check for existing instance and clean up if needed.
      if (this.instances.has(element)) {
        this.stop(element);
      }

      // Execute the library-specific loader.
      const activeLibrary = settings.countdown.activeLibrary;
      if (this.loaders[activeLibrary]) {
        this.loaders[activeLibrary](element, settings);
      }
      else {
        console.warn('Countdown: No loader found for library', activeLibrary);
      }
    });
  };

  /**
   * Stop a countdown timer instance.
   *
   * @param {Element} element
   *   The timer element to stop.
   */
  Drupal.countdown.stop = function(element) {
    const instance = this.instances.get(element);
    if (!instance) {
      return;
    }

    // Call appropriate stop method based on instance type.
    if (typeof instance.destroy === 'function') {
      instance.destroy();
    }
    else if (typeof instance.stop === 'function') {
      instance.stop();
    }
    else if (instance.counter && typeof instance.counter.stop === 'function') {
      instance.counter.stop();
    }

    // Clean up.
    this.instances.delete(element);
    element.classList.remove('countdown-initialized');
    element.innerHTML = '';

    // Remove once marker to allow re-initialization.
    delete element.dataset.onceCountdownInit;
  };

  /**
   * Restart a countdown timer instance.
   *
   * @param {Element} element
   *   The timer element to restart.
   */
  Drupal.countdown.restart = function(element) {
    const instance = this.instances.get(element);
    if (!instance) {
      return;
    }

    // Call appropriate restart method based on instance type.
    if (typeof instance.reset === 'function') {
      instance.reset();
    }
    else if (typeof instance.restart === 'function') {
      instance.restart();
    }
    else if (typeof instance.start === 'function') {
      instance.start();
    }
    else if (instance.counter && typeof instance.counter.start === 'function') {
      instance.counter.start();
    }
  };

  /**
   * Update the target date of a countdown timer.
   *
   * @param {Element} element
   *   The timer element to update.
   * @param {string} newTarget
   *   The new target date string.
   */
  Drupal.countdown.updateTarget = function(element, newTarget) {
    element.dataset.countdownTarget = newTarget;

    // Re-initialize the timer.
    this.stop(element);
    Drupal.behaviors.countdownIntegration.attach(element.parentNode, drupalSettings);
  };

  /**
   * Main countdown integration behavior.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countdownIntegration = {
    attach: function (context, settings) {
      // Validate settings exist.
      if (!settings.countdown || !settings.countdown.activeLibrary) {
        return;
      }

      const activeLibrary = settings.countdown.activeLibrary;

      // Load the library integration if needed.
      Drupal.countdown.loadIntegration(activeLibrary, function() {
        // Initialize all countdown timers.
        Drupal.countdown.initialize(context, settings);

        // Handle preview element specially.
        const previewElement = document.getElementById('countdown-preview');
        if (previewElement && context.contains(previewElement)) {
          // Clear any loading message.
          const loadingMessage = previewElement.querySelector('.countdown-preview-loading');
          if (loadingMessage) {
            loadingMessage.remove();
          }

          // Mark as preview-initialized.
          if (!previewElement.classList.contains('preview-initialized')) {
            previewElement.classList.add('preview-initialized');
          }
        }
      });
    }
  };

})(Drupal, drupalSettings, once);
