/**
 * @file
 * Integration for the core CountdownTimer library.
 *
 * This file handles initialization and management of the built-in
 * CountdownTimer instances that ship with the module.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize a core countdown timer.
   *
   * @param {Element} element
   *   The DOM element to initialize as a timer.
   * @param {Object} settings
   *   The settings object from drupalSettings.
   */
  function initializeCoreTimer(element, settings) {
    // Validate library availability.
    if (typeof CountdownTimer === 'undefined') {
      console.error('Countdown: Core timer library not loaded.');
      return;
    }

    // Extract configuration.
    const targetDate = element.dataset.countdownTarget;
    const elementSettings = element.dataset.countdownSettings ?
      JSON.parse(element.dataset.countdownSettings) : {};
    const librarySettings = settings.countdown.countdown || {};
    const finalSettings = Object.assign({}, librarySettings, elementSettings);

    // Ensure element has an ID for the selector option.
    if (!element.id) {
      element.id = 'countdown-' + Math.random().toString(36).substr(2, 9);
    }

    // Calculate time difference.
    const now = new Date().getTime();
    const target = new Date(targetDate).getTime();
    const diff = target - now;

    if (diff <= 0) {
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
      return;
    }

    // Convert to time components.
    const seconds = Math.floor(diff / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);

    // Build start string.
    let startString = '';
    if (days > 0) {
      startString = days + 'd ' + (hours % 24) + 'h ' + (minutes % 60) + 'm ' +
        (seconds % 60) + 's';
    }
    else if (hours > 0) {
      startString = String(hours).padStart(2, '0') + ':' +
        String(minutes % 60).padStart(2, '0') + ':' +
        String(seconds % 60).padStart(2, '0');
    }
    else {
      startString = String(minutes % 60).padStart(2, '0') + ':' +
        String(seconds % 60).padStart(2, '0');
    }

    // Create timer instance.
    const timer = new CountdownTimer({
      selector: '#' + element.id,
      mode: 'countdown',
      start: startString,
      precision: finalSettings.precision || 'seconds',
      autoStart: false,
      debug: finalSettings.debug || false,
      onComplete: function(timeObj, timerInstance) {
        element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
        if (typeof finalSettings.onComplete === 'function') {
          finalSettings.onComplete.call(this, element);
        }

        // Dispatch custom event.
        element.dispatchEvent(new CustomEvent('countdown:complete', {
          detail: { element: element, timeObj: timeObj }
        }));
      },
      onTick: function(timeObj, timerInstance) {
        // Format the display.
        const pad = function(n) { return n < 10 ? '0' + n : n; };

        let display = '';
        if (timeObj.days > 0) {
          display = timeObj.days + 'd ' + pad(timeObj.hours) + 'h ' +
            pad(timeObj.minutes) + 'm ' + pad(timeObj.seconds) + 's';
        }
        else if (timeObj.hours > 0) {
          display = pad(timeObj.hours) + ':' + pad(timeObj.minutes) + ':' +
            pad(timeObj.seconds);
        }
        else {
          display = pad(timeObj.minutes) + ':' + pad(timeObj.seconds);
        }

        element.innerHTML = '<div class="countdown-display">' + display + '</div>';
      }
    });

    // Start the timer.
    timer.start();

    // Store instance for management.
    Drupal.countdown.instances.set(element, timer);

    // Mark as initialized.
    element.classList.add('countdown-initialized');

    // Dispatch initialization event.
    element.dispatchEvent(new CustomEvent('countdown:initialized', {
      detail: {
        library: 'countdown',
        element: element,
        settings: finalSettings
      }
    }));
  }

  // Register the loader.
  Drupal.countdown.registerLoader('countdown', initializeCoreTimer);

})(Drupal);
