/**
 * @file
 * Integration for the FlipDown.js library.
 *
 * This file handles initialization and management of FlipDown instances
 * for countdown timers using the lightweight FlipDown.js library.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize a FlipDown timer.
   *
   * @param {Element} element
   *   The DOM element to initialize as a timer.
   * @param {Object} settings
   *   The settings object from drupalSettings.
   */
  function initializeFlipDown(element, settings) {
    // Validate library availability.
    if (typeof FlipDown === 'undefined') {
      console.error('Countdown: FlipDown library not loaded.');
      return;
    }

    // Extract configuration.
    const targetDate = element.dataset.countdownTarget;
    const elementSettings = element.dataset.countdownSettings ?
      JSON.parse(element.dataset.countdownSettings) : {};
    const librarySettings = settings.countdown.flipdown || {};
    const finalSettings = Object.assign({}, librarySettings, elementSettings);

    // Clean up any existing instance.
    if (Drupal.countdown.instances.has(element)) {
      element.innerHTML = '';
    }

    // Ensure element has unique ID.
    if (!element.id) {
      element.id = 'flipdown-' + Math.random().toString(36).substr(2, 9);
    }

    // Add required classes.
    element.classList.add('flipdown');
    const theme = finalSettings.theme || 'dark';
    element.classList.add('flipdown__theme-' + theme);

    // Convert to Unix timestamp in seconds.
    const timestamp = Math.floor(new Date(targetDate).getTime() / 1000);

    // Check if already expired.
    const now = Math.floor(Date.now() / 1000);
    if (timestamp <= now) {
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
      return;
    }

    // Clear element for clean initialization.
    element.innerHTML = '';

    // Create FlipDown instance.
    const flipdown = new FlipDown(timestamp, element.id, {
      theme: theme,
      headings: finalSettings.headings || ['Days', 'Hours', 'Minutes', 'Seconds']
    });

    // Start the timer.
    flipdown.start();

    // Add completion handler.
    flipdown.ifEnded(function() {
      // Handle completion callback.
      if (typeof finalSettings.onComplete === 'function') {
        finalSettings.onComplete.call(this, element);
      }

      // Update display if message provided.
      if (finalSettings.finishMessage) {
        element.innerHTML = finalSettings.finishMessage;
      }

      // Dispatch complete event.
      element.dispatchEvent(new CustomEvent('countdown:complete', {
        detail: { element: element, library: 'flipdown' }
      }));
    });

    // Store instance for management.
    Drupal.countdown.instances.set(element, flipdown);

    // Mark as initialized.
    element.classList.add('countdown-initialized');

    // Dispatch initialization event.
    element.dispatchEvent(new CustomEvent('countdown:initialized', {
      detail: {
        library: 'flipdown',
        element: element,
        settings: finalSettings
      }
    }));
  }

  // Register the loader.
  Drupal.countdown.registerLoader('flipdown', initializeFlipDown);

})(Drupal);
