<?php

namespace Drupal\countdown\Element;

use Drupal\Core\Render\Element\RenderElement;

/**
 * Provides a countdown meta information render element.
 *
 * @RenderElement("countdown_meta")
 */
class CountdownMeta extends RenderElement {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = get_class($this);
    return [
      '#pre_render' => [
        [$class, 'preRenderCountdownMeta'],
      ],
      '#theme' => 'countdown_meta',
      '#attributes' => [],
      '#form_state' => NULL,
    ];
  }

  /**
   * Pre-render callback for the countdown meta element.
   *
   * @param array $element
   *   The render element.
   *
   * @return array
   *   The modified render element.
   */
  public static function preRenderCountdownMeta(array $element) {
    // Ensure ID is set.
    if (empty($element['#attributes']['id'])) {
      $element['#attributes']['id'] = 'library-meta-wrapper';
    }

    // Get form state if available.
    $form_state = $element['#form_state'] ?? NULL;

    // Get configuration.
    $config = \Drupal::config('countdown.settings');

    // Build meta information.
    $meta = [];

    // Get current method.
    $method = NULL;
    if ($form_state) {
      $method = $form_state->getValue('method');
    }
    if ($method === NULL) {
      $method = $config->get('method') ?: 'local';
    }
    $meta['method'] = $method;

    // Get current library and plugin.
    $library_id = NULL;
    if ($form_state) {
      $library_id = $form_state->getValue('library');
    }
    if ($library_id === NULL) {
      $library_id = $config->get('library');
    }

    $meta['plugin_label'] = \Drupal::translation()->translate('None');
    $meta['installed'] = FALSE;
    $meta['path'] = NULL;
    $meta['version'] = NULL;

    if ($library_id && $library_id !== 'none') {
      $plugin_manager = \Drupal::service('plugin.manager.countdown_library');
      $plugin = $plugin_manager->getPlugin($library_id);

      if ($plugin) {
        $meta['plugin_label'] = $plugin->getLabel();

        if ($method === 'local') {
          $meta['installed'] = $plugin->isInstalled();
          if ($meta['installed']) {
            $meta['path'] = $plugin->getLibraryPath();
            $meta['version'] = $plugin->getInstalledVersion();
          }
        }
      }
    }

    // Get CDN provider if in CDN mode.
    $meta['cdn_provider'] = NULL;
    if ($method === 'cdn') {
      $cdn_provider = NULL;
      if ($form_state) {
        $cdn_provider = $form_state->getValue('cdn_provider');
      }
      if ($cdn_provider === NULL) {
        $cdn_provider = $config->get('cdn.provider') ?: 'jsdelivr';
      }
      $meta['cdn_provider'] = $cdn_provider;
    }

    // Get RTL setting.
    $rtl = NULL;
    if ($form_state) {
      $rtl = $form_state->getValue('rtl');
    }
    if ($rtl === NULL) {
      $rtl = $config->get('rtl') ?? FALSE;
    }
    $meta['rtl'] = (bool) $rtl;

    // Get debug mode.
    $debug = NULL;
    if ($form_state) {
      $debug = $form_state->getValue('debug_mode');
    }
    if ($debug === NULL) {
      $debug = $config->get('debug_mode') ?? FALSE;
    }
    $meta['debug'] = (bool) $debug;

    // Set the meta data.
    $element['#meta'] = $meta;

    return $element;
  }

}
