<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin\CountdownLibrary;

use Drupal\countdown\Plugin\CountdownLibraryPluginBase;

/**
 * PQINA Flip countdown library plugin implementation.
 *
 * Flip is a modern, high-performance countdown timer with smooth
 * animations and a clean, minimalist design. Part of the PQINA
 * suite of JavaScript components.
 *
 * @CountdownLibrary(
 *   id = "flip",
 *   label = @Translation("PQINA Flip"),
 *   description = @Translation("Modern timer with smooth flip animations"),
 *   type = "external",
 *   homepage = "https://pqina.nl/flip",
 *   repository = "https://github.com/pqina/flip",
 *   version = "1.8.4",
 *   npm_package = "@pqina/flip",
 *   folder_names = {
 *     "flip",
 *     "@pqina-flip",
 *     "pqina-flip",
 *     "flip-master",
 *     "pqina-flip-master"
 *   },
 *   init_function = "Flip",
 *   author = "PQINA",
 *   license = "MIT",
 *   dependencies = {
 *     "core/drupal",
 *     "core/once"
 *   },
 *   weight = 2,
 *   experimental = false,
 *   api_version = "1.0"
 * )
 */
final class Flip extends CountdownLibraryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getAssetMap(): array {
    return [
      'local' => [
        'js' => [
          'development' => 'dist/flip.js',
          'production' => 'dist/flip.min.js',
        ],
        'css' => [
          'development' => 'dist/flip.css',
          'production' => 'dist/flip.min.css',
        ],
      ],
      'cdn' => [
        'jsdelivr' => [
          'js' => '//cdn.jsdelivr.net/npm/@pqina/flip@1.8.4/dist/flip.min.js',
          'css' => '//cdn.jsdelivr.net/npm/@pqina/flip@1.8.4/dist/flip.min.css',
        ],
        'unpkg' => [
          'js' => '//unpkg.com/@pqina/flip@1.8.4/dist/flip.min.js',
          'css' => '//unpkg.com/@pqina/flip@1.8.4/dist/flip.min.css',
        ],
        'cdnjs' => [
          'js' => '//cdnjs.cloudflare.com/ajax/libs/flip/1.8.4/flip.min.js',
          'css' => '//cdnjs.cloudflare.com/ajax/libs/flip/1.8.4/flip.min.css',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDependencies(): array {
    return [
      'core/drupal',
      'core/once',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getHomepage(): ?string {
    return 'https://pqina.nl/flip';
  }

  /**
   * {@inheritdoc}
   */
  protected function detectVersionCustom(string $base_path): ?string {
    // Try multiple strategies to detect Flip version.
    // Strategy 1: Check the main JS file for version information.
    $js_files = [
      '/dist/flip.js',
      '/dist/flip.min.js',
      '/lib/flip.js',
      '/src/flip.js',
    ];

    foreach ($js_files as $js_file) {
      $file_path = $base_path . $js_file;

      if (file_exists($file_path)) {
        try {
          // Read first 10KB of file.
          $handle = fopen($file_path, 'r');
          $content = fread($handle, 10240);
          fclose($handle);

          // Look for version patterns specific to PQINA Flip.
          $patterns = [
            '/\*\s+Flip\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/\*\s+@pqina\/flip\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/\*\s+@version\s+([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/version["\']?\s*[:=]\s*["\']([0-9]+\.[0-9]+(?:\.[0-9]+)?)["\']/',
            '/Flip\.version\s*=\s*["\']([^"\']+)["\']/',
            '/FLIP_VERSION\s*=\s*["\']([0-9]+\.[0-9]+(?:\.[0-9]+)?)["\']/',
          ];

          foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
              $this->logger->info('PQINA Flip version detected: @version from @file', [
                '@version' => $matches[1],
                '@file' => $js_file,
              ]);
              return $this->normalizeVersion($matches[1]);
            }
          }
        }
        catch (\Exception $e) {
          $this->logger->error('Error reading Flip file @file: @message', [
            '@file' => $js_file,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    // Strategy 2: Check CSS file for version comments.
    $css_files = [
      '/dist/flip.css',
      '/dist/flip.min.css',
      '/lib/flip.css',
    ];

    foreach ($css_files as $css_file) {
      $file_path = $base_path . $css_file;

      if (file_exists($file_path)) {
        try {
          $handle = fopen($file_path, 'r');
          $content = fread($handle, 2048);
          fclose($handle);

          $pattern = '/(?:Flip|@pqina\/flip)\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/i';
          if (preg_match($pattern, $content, $matches)) {
            return $this->normalizeVersion($matches[1]);
          }
        }
        catch (\Exception $e) {
          $this->logger->warning('Could not read Flip CSS file: @file', [
            '@file' => $css_file,
          ]);
        }
      }
    }

    // Strategy 3: Check package.json specifically.
    $package_file = $base_path . '/package.json';
    if (file_exists($package_file)) {
      try {
        $content = file_get_contents($package_file);
        $package_data = json_decode($content, TRUE);

        if (json_last_error() === JSON_ERROR_NONE) {
          // Check if it's the right package.
          $name = $package_data['name'] ?? '';
          if ($name === '@pqina/flip' || $name === 'flip') {
            if (!empty($package_data['version'])) {
              return $this->normalizeVersion($package_data['version']);
            }
          }
        }
      }
      catch (\Exception $e) {
        $this->logger->warning('Could not read Flip package.json', [
          '@error' => $e->getMessage(),
        ]);
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInstallation(string $path): bool {
    // First check parent validation.
    if (!parent::validateInstallation($path)) {
      return FALSE;
    }

    // Additional Flip-specific validation.
    $path = ltrim($path, '/');
    $full_path = DRUPAL_ROOT . '/' . $path;

    // Check for PQINA Flip-specific indicators.
    $flip_indicators = [
      '/dist/flip.js',
      '/dist/flip.css',
      '/lib/flip.js',
      '/package.json',
    ];

    $found_indicator = FALSE;
    foreach ($flip_indicators as $indicator) {
      $check_path = $full_path . $indicator;

      if ($indicator === '/package.json' && file_exists($check_path)) {
        // Verify it's actually PQINA Flip.
        try {
          $content = file_get_contents($check_path);
          $package = json_decode($content, TRUE);

          $name = $package['name'] ?? '';
          if ($name === '@pqina/flip' || $name === 'flip') {
            $found_indicator = TRUE;
            break;
          }
        }
        catch (\Exception $e) {
          // Continue checking other indicators.
        }
      }
      elseif (file_exists($check_path)) {
        $found_indicator = TRUE;
        break;
      }
    }

    if (!$found_indicator) {
      $this->logger->warning('PQINA Flip library structure not recognized at @path', [
        '@path' => $path,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredFiles(): array {
    return [
      'dist/flip.js',
      'dist/flip.css',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAlternativePaths(): array {
    return [
      [
        'lib/flip.js',
        'lib/flip.css',
      ],
      [
        'build/flip.js',
        'build/flip.css',
      ],
      [
        'src/flip.js',
        'src/flip.css',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtensions(): bool {
    // Flip library doesn't have extensions.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildAttachments(array $config): array {
    // Get the base attachments from parent.
    $attachments = parent::buildAttachments($config);

    // Check once whether Tick is installed using discovery injected in base.
    $tick_installed = $this->libraryDiscovery->isInstalled('tick');

    if ($tick_installed) {
      $attachments['#attached']['library'][] = 'countdown/tick_flip';
    }

    return $attachments;
  }

}
