<?php

declare(strict_types=1);

namespace Drupal\countdown\Utility;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;

/**
 * Provides typed access to countdown configuration.
 *
 * This utility centralizes configuration access with type safety and default
 * values, reducing duplication across services.
 */
class ConfigAccessor {

  /**
   * The configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a ConfigAccessor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->config = $config_factory->get('countdown.settings');
  }

  /**
   * Gets the active library ID.
   *
   * @return string
   *   The active library ID or 'countdown' as default.
   */
  public function getActiveLibrary(): string {
    return (string) ($this->config->get('library') ?: 'countdown');
  }

  /**
   * Gets the loading method.
   *
   * @return string
   *   Either 'local' or 'cdn'.
   */
  public function getLoadingMethod(): string {
    $method = $this->config->get('method');
    return in_array($method, ['local', 'cdn']) ? $method : 'local';
  }

  /**
   * Gets the CDN provider.
   *
   * @return string
   *   The CDN provider name.
   */
  public function getCdnProvider(): string {
    return (string) ($this->config->get('cdn.provider') ?: 'jsdelivr');
  }

  /**
   * Checks if auto-loading is enabled.
   *
   * @return bool
   *   TRUE if auto-loading is enabled.
   */
  public function isAutoLoadEnabled(): bool {
    return (bool) $this->config->get('load');
  }

  /**
   * Gets the build variant setting.
   *
   * @return bool
   *   TRUE for production/minified, FALSE for development.
   */
  public function getBuildVariant(): bool {
    return (bool) $this->config->get('build.variant');
  }

  /**
   * Checks if RTL support is enabled.
   *
   * @return bool
   *   TRUE if RTL is enabled.
   */
  public function isRtlEnabled(): bool {
    return (bool) $this->config->get('rtl');
  }

  /**
   * Checks if debug mode is enabled.
   *
   * @return bool
   *   TRUE if debug mode is enabled.
   */
  public function isDebugMode(): bool {
    return (bool) $this->config->get('advanced.debug_mode');
  }

  /**
   * Gets the cache lifetime in seconds.
   *
   * @return int
   *   The cache lifetime.
   */
  public function getCacheLifetime(): int {
    return (int) ($this->config->get('advanced.cache_lifetime') ?: 86400);
  }

  /**
   * Gets theme visibility settings.
   *
   * @return array
   *   Array with 'themes' and 'negate' keys.
   */
  public function getThemeVisibility(): array {
    return [
      'themes' => $this->config->get('theme_groups.themes') ?: [],
      'negate' => (int) $this->config->get('theme_groups.negate'),
    ];
  }

  /**
   * Gets page visibility settings.
   *
   * @return array
   *   Array with 'pages' and 'negate' keys.
   */
  public function getPageVisibility(): array {
    return [
      'pages' => $this->config->get('request_path.pages') ?: [],
      'negate' => (int) $this->config->get('request_path.negate'),
    ];
  }

  /**
   * Gets extensions configuration for a library.
   *
   * @param string $library_id
   *   The library ID.
   *
   * @return array
   *   Array of enabled extension IDs.
   */
  public function getLibraryExtensions(string $library_id): array {
    $extensions = $this->config->get("file_assets.extensions.{$library_id}");
    return is_array($extensions) ? array_filter($extensions) : [];
  }

  /**
   * Gets the full configuration object.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   The configuration object.
   */
  public function getConfig(): ImmutableConfig {
    return $this->config;
  }

}
