/**
 * @file
 * Integration for the PQINA Flip library.
 *
 * This file handles initialization and management of PQINA Flip instances.
 * Note that Flip requires the Tick library as a dependency.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize a PQINA Flip timer.
   *
   * @param {Element} element
   *   The DOM element to initialize as a timer.
   * @param {Object} settings
   *   The settings object from drupalSettings.
   */
  function initializeFlip(element, settings) {
    // Validate library availability.
    if (typeof Tick === 'undefined' || !Tick || !Tick.DOM) {
      console.error('Countdown: PQINA Flip requires Tick to be loaded.');
      return;
    }

    // Extract configuration.
    const targetDate = element.dataset.countdownTarget;
    const elementSettings = element.dataset.countdownSettings ?
      JSON.parse(element.dataset.countdownSettings) : {};
    const librarySettings = settings.countdown.flip || {};
    const finalSettings = Object.assign({}, librarySettings, elementSettings);

    // Check if already expired.
    const now = Date.now();
    const target = new Date(targetDate).getTime();
    if (target <= now) {
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
      return;
    }

    // Build the Flip markup structure.
    element.innerHTML = [
      '<div class="tick">',
      '  <div data-repeat="true" data-layout="horizontal center fit"',
      '       data-transform="preset(d, h, m, s) -> delay">',
      '    <div class="tick-group">',
      '      <div data-key="value" data-repeat="true"',
      '           data-transform="pad(00) -> split -> delay">',
      '        <span data-view="flip"></span>',
      '      </div>',
      '      <span data-key="label" data-view="text" class="tick-label"></span>',
      '    </div>',
      '  </div>',
      '</div>',
    ].join('');

    // Create Tick DOM instance.
    const tickDom = Tick.DOM.create(element.querySelector('.tick'));

    // Create countdown counter.
    const counter = Tick.count.down(targetDate);

    // Handle updates.
    counter.onupdate = function (value) {
      tickDom.value = value;

      // Dispatch tick event.
      element.dispatchEvent(new CustomEvent('countdown:tick', {
        detail: { element: element, library: 'flip', value: value }
      }));
    };

    // Handle completion.
    counter.onended = function() {
      // Update display.
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';

      // Execute callback.
      if (typeof finalSettings.onComplete === 'function') {
        finalSettings.onComplete.call(this, element);
      }

      // Dispatch complete event.
      element.dispatchEvent(new CustomEvent('countdown:complete', {
        detail: { element: element, library: 'flip' }
      }));
    };

    // Store instance for management.
    Drupal.countdown.instances.set(element, counter);

    // Mark as initialized.
    element.classList.add('countdown-initialized');

    // Dispatch initialization event.
    element.dispatchEvent(new CustomEvent('countdown:initialized', {
      detail: {
        library: 'flip',
        element: element,
        settings: finalSettings
      }
    }));
  }

  // Register the loader.
  Drupal.countdown.registerLoader('flip', initializeFlip);

})(Drupal);
