/**
 * @file
 * Integration for the FlipClock.js library.
 *
 * This file handles initialization and management of FlipClock instances
 * for countdown timers using the FlipClock.js library.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize a FlipClock timer.
   *
   * @param {Element} element
   *   The DOM element to initialize as a timer.
   * @param {Object} settings
   *   The settings object from drupalSettings.
   */
  function initializeFlipClock(element, settings) {
    // Validate library availability.
    if (typeof FlipClock === 'undefined') {
      console.error('Countdown: FlipClock library not loaded.');
      return;
    }

    // Extract configuration.
    const elementSettings = element.dataset.countdownSettings
      ? JSON.parse(element.dataset.countdownSettings) : {};
    const libSettings = (settings.countdown && settings.countdown.flipclock) || {};
    const finalSettings = Object.assign({}, libSettings, elementSettings);

    // Calculate difference in seconds.
    const targetDateStr = element.dataset.countdownTarget;
    const targetDate = new Date(targetDateStr);

    // Map legacy names to real faces.
    const faceMap = {
      DailyCounter: 'DayCounter',
      HourlyCounter: 'HourCounter',
      MinuteCounter: 'MinuteCounter',
      '24h': 'TwentyFourHourClock',
      '12h': 'TwelveHourClock',
    };
    const requested = finalSettings.face || finalSettings.clockFace || 'DayCounter';
    const face = faceMap[requested] || requested;


    // Optional: colon divider via theme override (library-native).
    // Default theme = FlipClock.defaults.theme (Original).
    // We clone and override only Divider renderer.
    let theme = FlipClock.defaults.theme; // has Divider, faces, etc. :contentReference[oaicite:2]{index=2}
    if (finalSettings.divider === 'colon') {
      theme = Object.assign({}, theme, {
        Divider: function (el /*, instance */) {
          el.classList.add('flip-clock-divider');
          el.textContent = ':'; // real colon instead of two dots
        },
      });
    }
    // Handle expired timer.
    if (targetDate <= 0) {
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
      return;
    }

    // Create FlipClock instance.
    const clock = new FlipClock(element, targetDate, {
      countdown: true,
      face,
      theme,
      clockFace: finalSettings.clockFace || 'DailyCounter',
      showSeconds: finalSettings.showSeconds !== false,
      language: finalSettings.language || 'en',
      autoStart: finalSettings.autoStart !== false,
      callbacks: {
        start: function() {
          // Dispatch start event.
          element.dispatchEvent(new CustomEvent('countdown:start', {
            detail: { element: element, library: 'flipclock' }
          }));
        },
        stop: function() {
          // Handle completion.
          if (typeof finalSettings.onComplete === 'function') {
            finalSettings.onComplete.call(this, element);
          }

          // Update display.
          if (finalSettings.finishMessage) {
            element.innerHTML = finalSettings.finishMessage;
          }

          // Dispatch complete event.
          element.dispatchEvent(new CustomEvent('countdown:complete', {
            detail: { element: element, library: 'flipclock' }
          }));
        },
        interval: function() {
          // Dispatch tick event.
          element.dispatchEvent(new CustomEvent('countdown:tick', {
            detail: { element: element, library: 'flipclock' }
          }));
        }
      }
    });

    // Store instance for management.
    Drupal.countdown.instances.set(element, clock);

    // Mark as initialized.
    element.classList.add('countdown-initialized');

    // Dispatch initialization event.
    element.dispatchEvent(new CustomEvent('countdown:initialized', {
      detail: {
        library: 'flipclock',
        element: element,
        settings: finalSettings
      }
    }));
  }

  // Register the loader.
  Drupal.countdown.registerLoader('flipclock', initializeFlipClock);

})(Drupal);
