/**
 * @file
 * Integration for PQINA Tick countdown library.
 *
 * This file provides the integration layer between Drupal and the Tick
 * countdown library, handling initialization, configuration, and view management.
 */

(function (Drupal) {
  'use strict';

  /**
   * Resolve which Tick view to use based on settings and element attributes.
   *
   * @param {Object} settings
   *   The settings object.
   * @param {Element} element
   *   The countdown element.
   *
   * @return {string}
   *   The view name to use.
   */
  function resolveTickView(settings, element) {
    // Priority order:
    // 1. Element data attribute (data-tick-view).
    // 2. Settings view property.
    // 3. Extension-based detection.
    // 4. Default to 'text'.

    const allowed = ['text', 'line', 'flip', 'dots', 'boom', 'swap'];

    // Check element attribute first.
    if (element.dataset.tickView) {
      const view = element.dataset.tickView.toLowerCase();
      if (allowed.indexOf(view) !== -1) {
        return view;
      }
    }

    // Check settings.
    if (typeof settings.view === 'string' && settings.view) {
      let view = settings.view.trim().toLowerCase();

      // Remove 'view_' prefix if present.
      if (view.indexOf('view_') === 0) {
        view = view.substring(5);
      }

      // Validate the view.
      if (allowed.indexOf(view) !== -1) {
        return view;
      }
    }

    // Default fallback.
    return 'text';
  }

  /**
   * Build the Tick markup structure with proper boom audio attributes.
   *
   * @param {string} view
   *   The view name for the data-view attribute.
   * @param {Object} settings
   *   The settings object.
   *
   * @return {string}
   *   The HTML markup string.
   */
  function buildTickMarkup(view, settings) {
    // Ensure format is an array.
    let format = settings.format || ['d', 'h', 'm', 's'];

    // If format is a string, convert it to an array.
    if (typeof format === 'string') {
      // Handle format strings like 'd[d] h[h] m[m] s[s]'.
      format = format.match(/[dhms]/g) || ['d', 'h', 'm', 's'];
    }

    // Ensure format is an array.
    if (!Array.isArray(format)) {
      format = ['d', 'h', 'm', 's'];
    }

    const formatString = format.join(', ');

    // Base markup structure.
    let markup = [
      '<div class="tick" data-credits="false" data-did-init="countdownTickInit">',
      '  <div data-repeat="true" data-layout="horizontal center fit"',
      '       data-transform="preset(' + formatString + ') -> delay">',
      '    <div class="tick-group">',
      '      <div data-key="value" data-repeat="true"',
      '           data-transform="pad(00) -> split -> delay">',
    ];

    // Add the appropriate view element based on the selected view.
    if (view === 'boom') {
      let modulePath = drupalSettings.countdown.modulePath;

      // Add boom view with audio samples.
      markup.push('        <span data-view="boom" data-style="sample: url(' +
        modulePath + '/media/bell.m4a), url(' +
        modulePath + '/media/bell.ogg);"></span>');
    }
    else {
      // Standard views just need the data-view attribute.
      markup.push('        <span data-view="' + view + '"></span>');
    }

    // Complete the markup structure.
    markup.push(
      '      </div>',
      '      <span data-key="label" data-view="text" class="tick-label"></span>',
      '    </div>',
      '  </div>',
      '</div>'
    );

    return markup.join('');
  }

  /**
   * Initialize a PQINA Tick timer.
   *
   * @param {Element} element
   *   The DOM element to initialize as a timer.
   * @param {Object} settings
   *   The settings object from drupalSettings.
   */
  function initializeTick(element, settings) {
    // Validate library availability.
    if (typeof Tick === 'undefined' || !Tick || !Tick.DOM) {
      console.error('Countdown: Tick library not loaded or DOM not available.');
      return;
    }

    // Extract configuration.
    const targetDate = element.dataset.countdownTarget;
    if (!targetDate) {
      console.error('Countdown: No target date specified for Tick timer.');
      return;
    }

    // Merge settings from various sources.
    const elementSettings = element.dataset.countdownSettings ?
      JSON.parse(element.dataset.countdownSettings) : {};

    // Look for settings in drupalSettings.countdown.tick.
    const globalTickSettings = settings.countdown && settings.countdown.tick ?
      settings.countdown.tick : {};

    // Merge all settings together.
    const finalSettings = Object.assign({}, globalTickSettings, elementSettings);

    // Check if already expired.
    const now = Date.now();
    const target = new Date(targetDate).getTime();
    if (target <= now) {
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
      element.classList.add('countdown-expired');
      return;
    }

    // Resolve which view to use.
    const view = resolveTickView(finalSettings, element);

    // Log debug information if debug mode is enabled.
    if (finalSettings.debug) {
      console.log('Countdown Tick: Initializing with view:', view);
      console.log('Countdown Tick: Settings:', finalSettings);
      console.log('Countdown Tick: Extensions loaded:', finalSettings.extensions);
    }

    // Build markup with the selected view.
    element.innerHTML = buildTickMarkup(view, finalSettings);

    // Find the tick root element.
    const tickRoot = element.querySelector('.tick');
    if (!tickRoot) {
      console.error('Countdown: Failed to create Tick markup.');
      return;
    }

    // Initialize Tick DOM.
    let tickDom;
    try {
      tickDom = Tick.DOM.create(tickRoot);
    }
    catch (error) {
      // Fall back to text view if requested view fails.
      console.warn('Countdown: Tick view "' + view + '" not available, falling back to text.', error);

      // Rebuild with text view.
      element.innerHTML = buildTickMarkup('text', finalSettings);
      const fallbackRoot = element.querySelector('.tick');

      try {
        tickDom = Tick.DOM.create(fallbackRoot);
      }
      catch (fallbackError) {
        console.error('Countdown: Failed to initialize Tick even with text view.', fallbackError);
        return;
      }
    }

    // Ensure format is an array for countdown creation.
    let format = finalSettings.format || ['d', 'h', 'm', 's'];
    if (typeof format === 'string') {
      format = format.match(/[dhms]/g) || ['d', 'h', 'm', 's'];
    }
    if (!Array.isArray(format)) {
      format = ['d', 'h', 'm', 's'];
    }

    // Create countdown counter.
    const counter = Tick.count.down(targetDate, {
      format: format,
      interval: finalSettings.interval || 1000
    });

    // Handle counter updates.
    counter.onupdate = function(value) {
      // Update the DOM with the new value.
      tickDom.value = value;

      // Dispatch tick event for other scripts to listen to.
      element.dispatchEvent(new CustomEvent('countdown:tick', {
        detail: {
          element: element,
          library: 'tick',
          value: value,
          view: view
        }
      }));
    };

    // Handle countdown completion.
    counter.onended = function() {
      // Update display with finish message.
      element.innerHTML = finalSettings.finishMessage || 'Time\'s up!';
      element.classList.add('countdown-expired');

      // Execute callback if provided.
      if (typeof finalSettings.onComplete === 'function') {
        finalSettings.onComplete.call(this, element);
      }

      // Dispatch complete event.
      element.dispatchEvent(new CustomEvent('countdown:complete', {
        detail: {
          element: element,
          library: 'tick',
          view: view
        }
      }));
    };

    // Store instance for management.
    // Store both counter and DOM for proper cleanup.
    Drupal.countdown.instances.set(element, {
      counter: counter,
      dom: tickDom,
      view: view
    });

    // Mark as initialized.
    element.classList.add('countdown-initialized');
    element.classList.add('countdown-tick-' + view);
    element.setAttribute('data-tick-view-active', view);

    // Dispatch initialization event.
    element.dispatchEvent(new CustomEvent('countdown:initialized', {
      detail: {
        library: 'tick',
        element: element,
        settings: finalSettings,
        view: view
      }
    }));
  }

  // Register the loader with the main countdown system.
  Drupal.countdown.registerLoader('tick', initializeTick);

})(Drupal);
