<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin\CountdownLibrary;

use Drupal\countdown\Plugin\CountdownLibraryPluginBase;

/**
 * Core countdown timer library plugin implementation.
 *
 * This is the built-in countdown timer that ships with the module.
 * It provides a lightweight, high-performance countdown/countup timer
 * with drift compensation and no external dependencies.
 *
 * @CountdownLibrary(
 *   id = "countdown",
 *   label = @Translation("Countdown Timer"),
 *   description = @Translation("Built-in timer with drift compensation"),
 *   type = "core",
 *   homepage = "https://drupal.org/project/countdown",
 *   repository = "https://git.drupalcode.org/project/countdown",
 *   version = "1.0.0",
 *   init_function = "CountdownTimer",
 *   author = "Drupal Community",
 *   license = "GPL-2.0+",
 *   dependencies = {
 *     "core/drupal",
 *     "core/drupalSettings",
 *     "core/once"
 *   },
 *   weight = -10,
 *   experimental = false,
 *   api_version = "1.0"
 * )
 */
final class CoreTimer extends CountdownLibraryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function isInstalled(): bool {
    // Core library is always installed.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getLibraryPath(): ?string {
    // Core library files are in the module directory.
    $module_handler = \Drupal::moduleHandler();
    $module_path = $module_handler->getModule('countdown')->getPath();
    return '/' . $module_path;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInstallation(string $path): bool {
    // Core timer validation checks for required files.
    $module_handler = \Drupal::moduleHandler();
    $module_path = $module_handler->getModule('countdown')->getPath();

    // Check for required core timer files.
    $required_files = [
      '/js/countdown.timer.js',
      '/js/countdown.integration.js',
      '/css/countdown.timer.css',
    ];

    foreach ($required_files as $file) {
      $file_path = DRUPAL_ROOT . '/' . $module_path . $file;
      if (!file_exists($file_path)) {
        $this->logger->error('Core timer file missing: @file', [
          '@file' => $file,
        ]);
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function detectVersion(string $path): ?string {
    // Core library version is fixed and matches the module version.
    return $this->getRequiredVersion();
  }

  /**
   * {@inheritdoc}
   */
  protected function detectVersionCustom(string $base_path): ?string {
    // Core library version is defined in the module info file.
    $module_handler = \Drupal::moduleHandler();
    $module = $module_handler->getModule('countdown');

    if ($module) {
      $info = \Drupal::service('extension.list.module')
        ->getExtensionInfo('countdown');
      if (isset($info['version'])) {
        // Remove Drupal version prefix if present.
        $version = preg_replace('/^\d+\.x-/', '', $info['version']);
        return $this->normalizeVersion($version);
      }
    }

    // Fallback to default version.
    return '1.0.0';
  }

  /**
   * {@inheritdoc}
   */
  public function getInstalledVersion(): ?string {
    // Core library version is always the current version.
    return $this->detectVersionCustom('');
  }

  /**
   * {@inheritdoc}
   */
  public function versionMeetsRequirements(): bool {
    // Core library always meets requirements.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getAssetMap(): array {
    // Core library assets are defined in countdown.libraries.yml.
    // We return an empty map as the module handles core assets directly.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getInitFunction(): ?string {
    return 'CountdownTimer';
  }

  /**
   * {@inheritdoc}
   */
  public function getDependencies(): array {
    return [
      'core/drupal',
      'core/drupalSettings',
      'core/once',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getHomepage(): ?string {
    return 'https://drupal.org/project/countdown';
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredFiles(): array {
    // Core library files are always present.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getPossibleFolderNames(): array {
    // Core library doesn't need folder discovery.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): array {
    return [
      'installed' => TRUE,
      'version_status' => 'ok',
      'messages' => [
        $this->t('Core library is built-in and always available.'),
        $this->t('Version: @version', [
          '@version' => $this->getInstalledVersion(),
        ]),
        $this->t('High-performance timer with drift compensation.'),
        $this->t('No external dependencies required.'),
      ],
      'severity' => 'info',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isExperimental(): bool {
    // Core library is stable.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    // Core library has highest priority.
    return -10;
  }

  /**
   * {@inheritdoc}
   */
  public function resetCache(): void {
    // Core library doesn't need cache reset as it's always available.
    // But we'll clear Drupal's library cache for consistency.
    \Drupal::service('library.discovery')->clearCachedDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtensions(): bool {
    // Core library doesn't have extensions.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getExtensionGroups(): array {
    // No extension groups for core library.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableExtensions(): array {
    // No extensions for core library.
    return [];
  }

}
