<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin\CountdownLibrary;

use Drupal\countdown\Plugin\CountdownLibraryPluginBase;

/**
 * FlipClock library plugin implementation.
 *
 * FlipClock is a proper abstract object oriented clock and counter library
 * with a realistic flip animation effect. It provides various clock faces
 * and countdown/countup functionality.
 *
 * @CountdownLibrary(
 *   id = "flipclock",
 *   label = @Translation("FlipClock"),
 *   description = @Translation("Clock and counter with realistic flip effect"),
 *   type = "external",
 *   homepage = "https://flipclockjs.com",
 *   repository = "https://github.com/objectivehtml/FlipClock",
 *   version = "0.10.8",
 *   npm_package = "flipclock",
 *   folder_names = {
 *     "flipclock",
 *     "FlipClock",
 *     "flipclock.js",
 *     "FlipClock.js",
 *     "FlipClock-master",
 *     "objectivehtml-FlipClock"
 *   },
 *   init_function = "FlipClock",
 *   author = "Objective HTML, LLC",
 *   license = "MIT",
 *   dependencies = {
 *     "core/jquery",
 *     "core/drupal"
 *   },
 *   weight = 0,
 *   experimental = false,
 *   api_version = "1.0"
 * )
 */
final class FlipClock extends CountdownLibraryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getAssetMap(): array {
    return [
      'local' => [
        'js' => [
          'development' => 'dist/flipclock.js',
          'production' => 'dist/flipclock.min.js',
        ],
        'css' => [
          'development' => 'dist/flipclock.css',
          'production' => 'dist/flipclock.css',
        ],
      ],
      'cdn' => [
        'jsdelivr' => [
          'js' => '//cdn.jsdelivr.net/npm/flipclock@0.10.8/dist/flipclock.min.js',
          'css' => '//cdn.jsdelivr.net/npm/flipclock@0.10.8/dist/flipclock.css',
        ],
        'unpkg' => [
          'js' => '//unpkg.com/flipclock@0.10.8/dist/flipclock.min.js',
          'css' => '//unpkg.com/flipclock@0.10.8/dist/flipclock.css',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDependencies(): array {
    return [
      'core/jquery',
      'core/drupal',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getHomepage(): ?string {
    return 'https://flipclockjs.com';
  }

  /**
   * {@inheritdoc}
   */
  protected function detectVersionCustom(string $base_path): ?string {
    // Try multiple strategies to detect FlipClock version.
    // Strategy 1: Check main JS file for version string.
    $js_files = [
      '/dist/flipclock.js',
      '/dist/flipclock.min.js',
      '/compiled/flipclock.js',
      '/src/flipclock/js/flipclock.js',
    ];

    foreach ($js_files as $js_file) {
      $file_path = $base_path . $js_file;

      if (file_exists($file_path)) {
        try {
          // Read only first 5KB to avoid loading large files.
          $handle = fopen($file_path, 'r');
          $content = fread($handle, 5120);
          fclose($handle);

          // Look for various version patterns.
          $patterns = [
            '/FlipClock\.version\s*=\s*["\']([^"\']+)["\']/',
            '/\*\s+@version\s+([^\s\*]+)/',
            '/\*\s+Version:\s*([^\s\*]+)/',
            '/version["\']?\s*[:=]\s*["\']([^"\']+)["\']/',
          ];

          foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
              $this->logger->info('FlipClock version detected: @version from @file', [
                '@version' => $matches[1],
                '@file' => $js_file,
              ]);
              return $this->normalizeVersion($matches[1]);
            }
          }
        }
        catch (\Exception $e) {
          $this->logger->error('Error reading FlipClock file @file: @message', [
            '@file' => $js_file,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    // Strategy 2: Check README or CHANGELOG files.
    $doc_files = [
      '/README.md',
      '/CHANGELOG.md',
      '/CHANGELOG',
      '/VERSION',
    ];

    foreach ($doc_files as $doc_file) {
      $file_path = $base_path . $doc_file;

      if (file_exists($file_path)) {
        try {
          $content = file_get_contents($file_path);

          // Look for version in documentation.
          $pattern = '/(?:version|v)\s*([0-9]+\.[0-9]+(?:\.[0-9]+)?)/i';
          if (preg_match($pattern, $content, $matches)) {
            return $this->normalizeVersion($matches[1]);
          }
        }
        catch (\Exception $e) {
          $this->logger->warning('Could not read FlipClock documentation file: @file', [
            '@file' => $doc_file,
          ]);
        }
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInstallation(string $path): bool {
    // First check parent validation.
    if (!parent::validateInstallation($path)) {
      return FALSE;
    }

    // Additional FlipClock-specific validation.
    $path = ltrim($path, '/');
    $full_path = DRUPAL_ROOT . '/' . $path;

    // Check for FlipClock-specific structure.
    $flipclock_indicators = [
      '/dist/flipclock.js',
      '/dist/flipclock.css',
      '/compiled/flipclock.js',
      '/src/flipclock',
    ];

    $found_indicator = FALSE;
    foreach ($flipclock_indicators as $indicator) {
      $check_path = $full_path . $indicator;
      if (file_exists($check_path) || is_dir($check_path)) {
        $found_indicator = TRUE;
        break;
      }
    }

    if (!$found_indicator) {
      $this->logger->warning('FlipClock library structure not recognized at @path', [
        '@path' => $path,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredFiles(): array {
    return [
      'dist/flipclock.js',
      'dist/flipclock.css',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAlternativePaths(): array {
    return [
      [
        'compiled/flipclock.js',
        'compiled/flipclock.css',
      ],
      [
        'src/flipclock/js/flipclock.js',
        'src/flipclock/css/flipclock.css',
      ],
      [
        'build/flipclock.js',
        'build/flipclock.css',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtensions(): bool {
    // FlipClock library doesn't have extensions.
    return FALSE;
  }

}
