<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin\CountdownLibrary;

use Drupal\countdown\Plugin\CountdownLibraryPluginBase;

/**
 * FlipDown library plugin implementation.
 *
 * FlipDown is a lightweight, performant flip-style countdown timer
 * with a clean, modern design and smooth animations.
 *
 * @CountdownLibrary(
 *   id = "flipdown",
 *   label = @Translation("FlipDown"),
 *   description = @Translation("Lightweight flip-style timer with modern design"),
 *   type = "external",
 *   homepage = "https://pbutcher.uk/flipdown",
 *   repository = "https://github.com/PButcher/flipdown",
 *   version = "0.3.2",
 *   npm_package = "flipdown",
 *   folder_names = {
 *     "flipdown",
 *     "FlipDown",
 *     "flipdown.js",
 *     "flipdown-master",
 *     "PButcher-flipdown"
 *   },
 *   init_function = "FlipDown",
 *   author = "Peter Butcher",
 *   license = "MIT",
 *   dependencies = {
 *     "core/drupal",
 *     "core/once"
 *   },
 *   weight = 1,
 *   experimental = false,
 *   api_version = "1.0"
 * )
 */
final class FlipDown extends CountdownLibraryPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getAssetMap(): array {
    return [
      'local' => [
        'js' => [
          'development' => 'dist/flipdown.js',
          'production' => 'dist/flipdown.min.js',
        ],
        'css' => [
          'development' => 'dist/flipdown.css',
          'production' => 'dist/flipdown.min.css',
        ],
      ],
      'cdn' => [
        'jsdelivr' => [
          'js' => '//cdn.jsdelivr.net/npm/flipdown@0.3.2/dist/flipdown.min.js',
          'css' => '//cdn.jsdelivr.net/npm/flipdown@0.3.2/dist/flipdown.min.css',
        ],
        'unpkg' => [
          'js' => '//unpkg.com/flipdown@0.3.2/dist/flipdown.min.js',
          'css' => '//unpkg.com/flipdown@0.3.2/dist/flipdown.min.css',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDependencies(): array {
    return [
      'core/drupal',
      'core/once',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getHomepage(): ?string {
    return 'https://pbutcher.uk/flipdown';
  }

  /**
   * {@inheritdoc}
   */
  protected function detectVersionCustom(string $base_path): ?string {
    // Try multiple strategies to detect FlipDown version.
    // Strategy 1: Check the main JS file for version information.
    $js_files = [
      '/dist/flipdown.js',
      '/dist/flipdown.min.js',
      '/src/flipdown.js',
      '/example/js/flipdown/flipdown.js',
    ];

    foreach ($js_files as $js_file) {
      $file_path = $base_path . $js_file;

      if (file_exists($file_path)) {
        try {
          // Read first 10KB of file.
          $handle = fopen($file_path, 'r');
          $content = fread($handle, 10240);
          fclose($handle);

          // Look for version patterns specific to FlipDown.
          $patterns = [
            '/\*\s+FlipDown\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/\*\s+@version\s+([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/version["\']?\s*[:=]\s*["\']([0-9]+\.[0-9]+(?:\.[0-9]+)?)["\']/',
            '/FlipDown\.version\s*=\s*["\']([^"\']+)["\']/',
            '/const\s+version\s*=\s*["\']([0-9]+\.[0-9]+(?:\.[0-9]+)?)["\']/',
          ];

          foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
              $this->logger->info('FlipDown version detected: @version from @file', [
                '@version' => $matches[1],
                '@file' => $js_file,
              ]);
              return $this->normalizeVersion($matches[1]);
            }
          }
        }
        catch (\Exception $e) {
          $this->logger->error('Error reading FlipDown file @file: @message', [
            '@file' => $js_file,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    // Strategy 2: Check CSS file for version comments.
    $css_files = [
      '/dist/flipdown.css',
      '/dist/flipdown.min.css',
      '/src/flipdown.css',
    ];

    foreach ($css_files as $css_file) {
      $file_path = $base_path . $css_file;

      if (file_exists($file_path)) {
        try {
          $handle = fopen($file_path, 'r');
          $content = fread($handle, 2048);
          fclose($handle);

          $pattern = '/FlipDown\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/i';
          if (preg_match($pattern, $content, $matches)) {
            return $this->normalizeVersion($matches[1]);
          }
        }
        catch (\Exception $e) {
          $this->logger->warning('Could not read FlipDown CSS file: @file', [
            '@file' => $css_file,
          ]);
        }
      }
    }

    // Strategy 3: Check documentation files.
    $doc_files = [
      '/README.md',
      '/CHANGELOG.md',
      '/VERSION',
      '/.version',
    ];

    foreach ($doc_files as $doc_file) {
      $file_path = $base_path . $doc_file;

      if (file_exists($file_path)) {
        try {
          $content = file_get_contents($file_path);

          // Look for version patterns.
          $pattern = '/(?:version|v)\s*([0-9]+\.[0-9]+(?:\.[0-9]+)?)/i';
          if (preg_match($pattern, $content, $matches)) {
            return $this->normalizeVersion($matches[1]);
          }
        }
        catch (\Exception $e) {
          $this->logger->warning('Could not read FlipDown documentation: @file', [
            '@file' => $doc_file,
          ]);
        }
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInstallation(string $path): bool {
    // First check parent validation.
    if (!parent::validateInstallation($path)) {
      return FALSE;
    }

    // Additional FlipDown-specific validation.
    $path = ltrim($path, '/');
    $full_path = DRUPAL_ROOT . '/' . $path;

    // Check for FlipDown-specific indicators.
    $flipdown_indicators = [
      '/dist/flipdown.js',
      '/dist/flipdown.css',
      '/src/flipdown.js',
      '/example/js/flipdown',
    ];

    $found_indicator = FALSE;
    foreach ($flipdown_indicators as $indicator) {
      $check_path = $full_path . $indicator;
      if (file_exists($check_path) || is_dir($check_path)) {
        $found_indicator = TRUE;
        break;
      }
    }

    if (!$found_indicator) {
      $this->logger->warning('FlipDown library structure not recognized at @path', [
        '@path' => $path,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredFiles(): array {
    return [
      'dist/flipdown.js',
      'dist/flipdown.css',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAlternativePaths(): array {
    return [
      [
        'src/flipdown.js',
        'src/flipdown.css',
      ],
      [
        'example/js/flipdown/flipdown.js',
        'example/css/flipdown/flipdown.css',
      ],
      [
        'lib/flipdown.js',
        'lib/flipdown.css',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtensions(): bool {
    // FlipDown library doesn't have extensions.
    return FALSE;
  }

}
