<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin\CountdownLibrary;

use Drupal\countdown\Plugin\CountdownLibraryPluginBase;

/**
 * PQINA Tick Counter library plugin implementation.
 *
 * Tick is a highly customizable counter with multiple views, themes,
 * and extensive configuration options. It supports various display
 * modes and animations through modular extensions.
 *
 * @CountdownLibrary(
 *   id = "tick",
 *   label = @Translation("PQINA Tick Counter"),
 *   description = @Translation("Highly customizable counter with multiple views, themes, and animation styles"),
 *   type = "external",
 *   homepage = "https://pqina.nl/tick",
 *   repository = "https://github.com/pqina/tick",
 *   version = "1.8.3",
 *   npm_package = "@pqina/tick",
 *   folder_names = {
 *     "tick",
 *     "@pqina-tick",
 *     "pqina-tick",
 *     "tick-master",
 *     "pqina-tick-master"
 *   },
 *   files = {
 *     "css" = {
 *       "development" = "dist/core/tick.core.css",
 *       "production" = "dist/core/tick.core.min.css"
 *     },
 *     "js" = {
 *       "development" = "dist/core/tick.core.global.js",
 *       "production" = "dist/core/tick.core.global.min.js"
 *     }
 *   },
 *   required_files = {
 *     "dist/core/tick.core.global.js",
 *     "dist/core/tick.core.css"
 *   },
 *   alternative_paths = {
 *     {
 *       "dist/tick.js",
 *       "dist/tick.css"
 *     },
 *     {
 *       "lib/tick.js",
 *       "lib/tick.css"
 *     },
 *     {
 *       "build/tick.core.js",
 *       "build/tick.core.css"
 *     }
 *   },
 *   init_function = "Tick",
 *   author = "PQINA",
 *   license = "MIT",
 *   dependencies = {},
 *   cdn = {
 *     "jsdelivr" = {
 *       "css" = "//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/core/tick.core.min.css",
 *       "js" = "//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/core/tick.core.global.min.js"
 *     },
 *     "unpkg" = {
 *       "css" = "//unpkg.com/@pqina/tick@1.8.3/dist/core/tick.core.min.css",
 *       "js" = "//unpkg.com/@pqina/tick@1.8.3/dist/core/tick.core.global.min.js"
 *     },
 *     "cdnjs" = {
 *       "css" = "//cdnjs.cloudflare.com/ajax/libs/tick/1.8.3/tick.core.min.css",
 *       "js" = "//cdnjs.cloudflare.com/ajax/libs/tick/1.8.3/tick.core.global.min.js"
 *     }
 *   },
 *   weight = 3,
 *   experimental = false,
 *   api_version = "1.0"
 * )
 */
final class Tick extends CountdownLibraryPluginBase {

  /**
   * Available Tick extensions with their configurations.
   *
   * @var array
   */
  protected const EXTENSIONS = [
    'font_highres' => [
      'label' => 'High Resolution Font',
      'description' => 'High-quality font rendering for displays',
      'files' => [
        'development' => 'dist/font-highres/tick.font.highres.global.js',
        'production' => 'dist/font-highres/tick.font.highres.global.min.js',
      ],
      'cdn' => [
        'jsdelivr' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/font-highres/tick.font.highres.global.min.js',
        'unpkg' => '//unpkg.com/@pqina/tick@1.8.3/dist/font-highres/tick.font.highres.global.min.js',
      ],
    ],
    'font_lowres' => [
      'label' => 'Low Resolution Font',
      'description' => 'Optimized font for smaller displays',
      'files' => [
        'development' => 'dist/font-lowres/tick.font.lowres.global.js',
        'production' => 'dist/font-lowres/tick.font.lowres.global.min.js',
      ],
      'cdn' => [
        'jsdelivr' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/font-lowres/tick.font.lowres.global.min.js',
        'unpkg' => '//unpkg.com/@pqina/tick@1.8.3/dist/font-lowres/tick.font.lowres.global.min.js',
      ],
    ],
    'view_boom' => [
      'label' => 'Boom View',
      'description' => 'Explosive animation effect',
      'css' => [
        'development' => 'dist/view-boom/tick.view.boom.css',
        'production' => 'dist/view-boom/tick.view.boom.min.css',
      ],
      'js' => [
        'development' => 'dist/view-boom/tick.view.boom.global.js',
        'production' => 'dist/view-boom/tick.view.boom.global.min.js',
      ],
      'cdn' => [
        'jsdelivr' => [
          'css' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-boom/tick.view.boom.min.css',
          'js' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-boom/tick.view.boom.global.min.js',
        ],
      ],
    ],
    'view_dots' => [
      'label' => 'Dots View',
      'description' => 'Dot matrix display style',
      'css' => [
        'development' => 'dist/view-dots/tick.view.dots.css',
        'production' => 'dist/view-dots/tick.view.dots.min.css',
      ],
      'js' => [
        'development' => 'dist/view-dots/tick.view.dots.global.js',
        'production' => 'dist/view-dots/tick.view.dots.global.min.js',
      ],
      'cdn' => [
        'jsdelivr' => [
          'css' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-dots/tick.view.dots.min.css',
          'js' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-dots/tick.view.dots.global.min.js',
        ],
      ],
    ],
    'view_line' => [
      'label' => 'Line View',
      'description' => 'Linear progress indicator',
      'css' => [
        'development' => 'dist/view-line/tick.view.line.css',
        'production' => 'dist/view-line/tick.view.line.min.css',
      ],
      'js' => [
        'development' => 'dist/view-line/tick.view.line.global.js',
        'production' => 'dist/view-line/tick.view.line.global.min.js',
      ],
      'cdn' => [
        'jsdelivr' => [
          'css' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-line/tick.view.line.min.css',
          'js' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-line/tick.view.line.global.min.js',
        ],
      ],
    ],
    'view_swap' => [
      'label' => 'Swap View',
      'description' => 'Smooth swapping animation',
      'css' => [
        'development' => 'dist/view-swap/tick.view.swap.css',
        'production' => 'dist/view-swap/tick.view.swap.min.css',
      ],
      'js' => [
        'development' => 'dist/view-swap/tick.view.swap.global.js',
        'production' => 'dist/view-swap/tick.view.swap.global.min.js',
      ],
      'cdn' => [
        'jsdelivr' => [
          'css' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-swap/tick.view.swap.min.css',
          'js' => '//cdn.jsdelivr.net/npm/@pqina/tick@1.8.3/dist/view-swap/tick.view.swap.global.min.js',
        ],
      ],
    ],
  ];

  /**
   * {@inheritdoc}
   */
  protected function detectVersionCustom(string $base_path): ?string {
    // Strategy 1: Check the main JS file for version information.
    $js_files = [
      '/dist/core/tick.core.global.js',
      '/dist/core/tick.core.global.min.js',
      '/dist/tick.js',
      '/lib/tick.js',
    ];

    // Try multiple strategies to detect Tick version.
    foreach ($js_files as $js_file) {
      $file_path = $base_path . $js_file;

      if (file_exists($file_path)) {
        try {
          // Read first 10KB of file.
          $handle = fopen($file_path, 'r');
          $content = fread($handle, 10240);
          fclose($handle);

          // Look for version patterns specific to PQINA Tick.
          $patterns = [
            '/\*\s+Tick\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/\*\s+@pqina\/tick\s+v?([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/\*\s+@version\s+([0-9]+\.[0-9]+(?:\.[0-9]+)?)/',
            '/Tick\.version\s*=\s*["\']([^"\']+)["\']/',
            '/TICK_VERSION\s*=\s*["\']([0-9]+\.[0-9]+(?:\.[0-9]+)?)["\']/',
            '/define\([\'"]version[\'"]\s*,\s*[\'"]([0-9]+\.[0-9]+(?:\.[0-9]+)?)[\'"]/',
          ];

          foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
              $this->logger->info('PQINA Tick version detected: @version from @file', [
                '@version' => $matches[1],
                '@file' => $js_file,
              ]);
              return $this->normalizeVersion($matches[1]);
            }
          }
        }
        catch (\Exception $e) {
          $this->logger->error('Error reading Tick file @file: @message', [
            '@file' => $js_file,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    // Strategy 2: Check package.json for @pqina/tick.
    $package_file = $base_path . '/package.json';
    if (file_exists($package_file)) {
      try {
        $content = file_get_contents($package_file);
        $package_data = json_decode($content, TRUE);

        if (json_last_error() === JSON_ERROR_NONE) {
          // Check if it's the right package.
          if (isset($package_data['name']) &&
              ($package_data['name'] === '@pqina/tick' || $package_data['name'] === 'tick')) {
            if (!empty($package_data['version'])) {
              return $this->normalizeVersion($package_data['version']);
            }
          }
        }
      }
      catch (\Exception $e) {
        $this->logger->warning('Could not read Tick package.json', [
          '@error' => $e->getMessage(),
        ]);
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getLibrarySettings(): array {
    $settings = parent::getLibrarySettings();

    // Add Tick-specific settings.
    $settings['view'] = 'swap';
    $settings['format'] = ['d', 'h', 'm', 's'];
    $settings['countdown'] = TRUE;
    $settings['duration'] = 1000;
    $settings['interval'] = 1000;
    $settings['autostart'] = TRUE;
    $settings['repeat'] = FALSE;
    $settings['presenters'] = [];

    // Get enabled extensions.
    $enabled_extensions = $this->getEnabledExtensions();
    $settings['extensions'] = $enabled_extensions;

    // Add available extensions for reference.
    $settings['availableExtensions'] = array_keys(self::EXTENSIONS);

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getAssets(bool $minified = TRUE): array {
    $assets = parent::getAssets($minified);

    // Add enabled extensions.
    $enabled_extensions = $this->getEnabledExtensions();
    $variant = $minified ? 'production' : 'development';
    $path = $this->getLibraryPath();

    // Check if we're using CDN.
    $config = \Drupal::config('countdown.settings');
    $method = $config->get('method') ?: 'local';
    $cdn_provider = $config->get('cdn.provider') ?: 'jsdelivr';

    foreach ($enabled_extensions as $ext_id) {
      if (isset(self::EXTENSIONS[$ext_id])) {
        $extension = self::EXTENSIONS[$ext_id];

        if ($method === 'cdn' && isset($extension['cdn'])) {
          // Use CDN URLs.
          if (isset($extension['cdn'][$cdn_provider])) {
            $cdn_config = $extension['cdn'][$cdn_provider];

            // Add CDN JS.
            if (is_array($cdn_config) && isset($cdn_config['js'])) {
              $assets['js'][] = [
                'path' => $cdn_config['js'],
                'external' => TRUE,
                'minified' => TRUE,
                'preprocess' => FALSE,
                'attributes' => ['crossorigin' => 'anonymous'],
              ];
            }
            elseif (is_string($cdn_config)) {
              // Single JS URL.
              $assets['js'][] = [
                'path' => $cdn_config,
                'external' => TRUE,
                'minified' => TRUE,
                'preprocess' => FALSE,
                'attributes' => ['crossorigin' => 'anonymous'],
              ];
            }

            // Add CDN CSS if present.
            if (is_array($cdn_config) && isset($cdn_config['css'])) {
              $assets['css'][] = [
                'path' => $cdn_config['css'],
                'external' => TRUE,
                'minified' => TRUE,
                'preprocess' => FALSE,
              ];
            }
          }
        }
        // Use local files.
        else {
          // Add JS files.
          if (isset($extension['files'][$variant])) {
            $file_path = $path . '/' . $extension['files'][$variant];
            if (file_exists(DRUPAL_ROOT . '/' . $file_path)) {
              $assets['js'][] = [
                'path' => $file_path,
                'external' => FALSE,
                'minified' => $minified,
                'preprocess' => $minified,
                'weight' => 10,
              ];
            }
          }
          elseif (isset($extension['js'][$variant])) {
            $file_path = $path . '/' . $extension['js'][$variant];
            if (file_exists(DRUPAL_ROOT . '/' . $file_path)) {
              $assets['js'][] = [
                'path' => $file_path,
                'external' => FALSE,
                'minified' => $minified,
                'preprocess' => $minified,
                'weight' => 10,
              ];
            }
          }

          // Add CSS files.
          if (isset($extension['css'][$variant])) {
            $file_path = $path . '/' . $extension['css'][$variant];
            if (file_exists(DRUPAL_ROOT . '/' . $file_path)) {
              $assets['css'][] = [
                'path' => $file_path,
                'external' => FALSE,
                'minified' => $minified,
                'preprocess' => $minified,
                'weight' => 10,
              ];
            }
          }
        }
      }
    }

    return $assets;
  }

  /**
   * Gets enabled Tick extensions from configuration.
   *
   * @return array
   *   Array of enabled extension IDs.
   */
  protected function getEnabledExtensions(): array {
    // Use the global Drupal config service directly.
    $config = \Drupal::config('countdown.settings');
    $extensions = $config->get('file_assets.extensions.tick');

    if (is_array($extensions)) {
      return array_filter($extensions);
    }

    // Default: enable swap view.
    return ['view_swap'];
  }

  /**
   * Gets available Tick extensions.
   *
   * @return array
   *   Array of extension information.
   */
  public function getAvailableExtensions(): array {
    return self::EXTENSIONS;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInstallation(string $path): bool {
    // First check parent validation.
    if (!parent::validateInstallation($path)) {
      return FALSE;
    }

    // Additional Tick-specific validation.
    $path = ltrim($path, '/');
    $full_path = DRUPAL_ROOT . '/' . $path;

    // Check for PQINA Tick-specific indicators.
    $tick_indicators = [
      '/dist/core/tick.core.global.js',
      '/dist/core/tick.core.css',
      '/dist/tick.js',
      '/package.json',
    ];

    $found_indicator = FALSE;
    foreach ($tick_indicators as $indicator) {
      $check_path = $full_path . $indicator;

      if ($indicator === '/package.json' && file_exists($check_path)) {
        // Verify it's actually PQINA Tick.
        try {
          $content = file_get_contents($check_path);
          $package = json_decode($content, TRUE);

          if (isset($package['name']) &&
              ($package['name'] === '@pqina/tick' || $package['name'] === 'tick')) {
            $found_indicator = TRUE;
            break;
          }
        }
        catch (\Exception $e) {
          // Continue checking other indicators.
        }
      }
      elseif (file_exists($check_path) || is_dir($check_path)) {
        $found_indicator = TRUE;
        break;
      }
    }

    if (!$found_indicator) {
      $this->logger->warning('PQINA Tick library structure not recognized at @path', [
        '@path' => $path,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildLibraryDefinition(bool $minified = TRUE): array {
    $definition = parent::buildLibraryDefinition($minified);

    // Tick doesn't require jQuery but benefits from Drupal behaviors.
    if (!in_array('core/drupal', $definition['dependencies'])) {
      $definition['dependencies'][] = 'core/drupal';
    }

    // Add once for safe initialization.
    if (!in_array('core/once', $definition['dependencies'])) {
      $definition['dependencies'][] = 'core/once';
    }

    return $definition;
  }

  /**
   * Validates that required extensions are available.
   *
   * @param array $extensions
   *   Array of extension IDs to validate.
   *
   * @return array
   *   Array of validation messages.
   */
  public function validateExtensions(array $extensions): array {
    $messages = [];
    $path = $this->getLibraryPath();

    if (!$path) {
      $messages[] = $this->t('Tick library is not installed.');
      return $messages;
    }

    foreach ($extensions as $ext_id) {
      if (!isset(self::EXTENSIONS[$ext_id])) {
        $messages[] = $this->t('Unknown Tick extension: @ext', ['@ext' => $ext_id]);
        continue;
      }

      $extension = self::EXTENSIONS[$ext_id];
      $found = FALSE;

      // Check for extension files.
      foreach (['files', 'js'] as $key) {
        if (isset($extension[$key])) {
          foreach ($extension[$key] as $file) {
            if (file_exists(DRUPAL_ROOT . '/' . $path . '/' . $file)) {
              $found = TRUE;
              break 2;
            }
          }
        }
      }

      if (!$found) {
        $messages[] = $this->t('Tick extension @ext not found in library.', [
          '@ext' => $extension['label'],
        ]);
      }
    }

    return $messages;
  }

  /**
   * {@inheritdoc}
   */
  public function getExtensionGroups(): array {
    return [
      'views' => [
        'label' => $this->t('Views'),
        'description' => $this->t('Different animation styles for the counter'),
        'extensions' => ['view_boom', 'view_dots', 'view_line', 'view_swap'],
        'weight' => 0,
      ],
      'fonts' => [
        'label' => $this->t('Fonts'),
        'description' => $this->t('Font rendering options'),
        'extensions' => ['font_highres', 'font_lowres'],
        'weight' => 10,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtensions(): bool {
    return TRUE;
  }

  /**
   * Build CDN definition with extensions support.
   *
   * @param bool $minified
   *   Whether to use minified assets.
   * @param string $provider
   *   The CDN provider to use.
   *
   * @return array
   *   The CDN library definition.
   */
  public function buildCdnDefinition(bool $minified = TRUE, string $provider = 'jsdelivr'): array {
    // Get the parent definition first.
    $definition = parent::buildCdnDefinition($minified, $provider);

    // Add enabled extensions from CDN.
    $enabled_extensions = $this->getEnabledExtensions();

    foreach ($enabled_extensions as $ext_id) {
      if (isset(self::EXTENSIONS[$ext_id]['cdn'][$provider])) {
        $ext_cdn = self::EXTENSIONS[$ext_id]['cdn'][$provider];

        // Add extension JS.
        if (isset($ext_cdn['js']) || is_string($ext_cdn)) {
          $js_url = is_string($ext_cdn) ? $ext_cdn : $ext_cdn['js'];
          $definition['js'][$js_url] = [
            'type' => 'external',
            'minified' => TRUE,
            'attributes' => [
              'defer' => TRUE,
              'crossorigin' => 'anonymous',
            ],
            'weight' => 10,
          ];
        }

        // Add extension CSS.
        if (is_array($ext_cdn) && isset($ext_cdn['css'])) {
          $definition['css']['theme'][$ext_cdn['css']] = [
            'type' => 'external',
            'minified' => TRUE,
            'weight' => 10,
          ];
        }
      }
    }

    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function buildAttachments(array $config): array {
    // Get the base attachments from parent.
    $attachments = parent::buildAttachments($config);

    // Add Tick extensions if enabled. We need to ensure font extensions are
    // loaded when dots view is selected, as dots view depends on the fonts.
    $enabled_extensions = $this->getEnabledExtensions();

    if (!empty($enabled_extensions)) {
      // Map extension IDs to library suffixes precisely.
      $extension_map = [
        'view_boom' => 'boom',
        'view_dots' => 'dots',
        'view_line' => 'line',
        'view_swap' => 'swap',
        'font_highres' => 'font_highres',
        'font_lowres' => 'font_lowres',
      ];

      // Automatically include font_highres when dots is selected as dots
      // view requires the font extension to work properly.
      if (in_array('view_dots', $enabled_extensions) && !in_array('font_highres', $enabled_extensions)) {
        $enabled_extensions[] = 'font_highres';
      }

      // Get the library discovery service to check library existence.
      $library_discovery = \Drupal::service('library.discovery');

      foreach ($enabled_extensions as $ext_id) {
        // Skip if extension is not recognized.
        if (!isset($extension_map[$ext_id])) {
          if ($config['debug'] ?? FALSE) {
            $this->logger->debug('Unknown Tick extension ID: @ext', [
              '@ext' => $ext_id,
            ]);
          }
          continue;
        }

        // Build the library name for this extension.
        $library_suffix = $extension_map[$ext_id];
        $library_name = 'tick_' . $library_suffix;

        // Add CDN suffix if using CDN method.
        if ($config['method'] === 'cdn') {
          $library_name .= '_cdn';
        }

        // Add minified suffix if using production variant.
        if ($config['variant'] ?? TRUE) {
          $library_name .= '.min';
        }

        // Build the full library identifier.
        $full_library_name = 'countdown/' . $library_name;

        // Check if library definition exists before attaching.
        $library_definition = $library_discovery->getLibraryByName('countdown', $library_name);

        if ($library_definition) {
          // Avoid duplicates.
          if (!in_array($full_library_name, $attachments['#attached']['library'])) {
            $attachments['#attached']['library'][] = $full_library_name;
          }
        }
        elseif ($config['debug'] ?? FALSE) {
          // Log missing library in debug mode.
          $this->logger->debug('Tick extension library not found: @lib', [
            '@lib' => $full_library_name,
          ]);
        }
      }
    }

    // Add the extensions to drupalSettings for the integration to use.
    if (!empty($enabled_extensions)) {
      // Normalize extension IDs for JavaScript.
      $normalized_extensions = [];
      foreach ($enabled_extensions as $ext_id) {
        // Remove 'view_' prefix for JavaScript.
        $normalized = str_starts_with($ext_id, 'view_') ? substr($ext_id, 5) : $ext_id;
        $normalized_extensions[] = $normalized;
      }

      $attachments['#attached']['drupalSettings']['countdown']['tick']['extensions'] = $normalized_extensions;

      // Determine the active view.
      $view = 'text';
      foreach ($enabled_extensions as $ext_id) {
        if (str_starts_with($ext_id, 'view_')) {
          $view = substr($ext_id, 5);
          break;
        }
      }
      $attachments['#attached']['drupalSettings']['countdown']['tick']['view'] = $view;

      // Add debug and flip settings.
      $attachments['#attached']['drupalSettings']['countdown']['tick']['debug'] = $config['debug'] ?? FALSE;
      $attachments['#attached']['drupalSettings']['countdown']['tick']['flip'] = $config['rtl'] ?? FALSE;
    }

    return $attachments;
  }

  /**
   * {@inheritdoc}
   */
  public function getType(): string {
    return 'external';
  }

  /**
   * {@inheritdoc}
   */
  public function getInitFunction(): string {
    return 'Tick';
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredAssets(): array {
    return [
      'js' => ['dist/core/tick.core.global.min.js'],
      'css' => ['dist/core/tick.core.min.css'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getVersion(): ?string {
    // Try to detect version from package.json file.
    $library_path = $this->getLibraryPath();
    if ($library_path) {
      $version = $this->detectVersionFromPackageJson($library_path);
      if ($version) {
        return $version;
      }
    }

    // Fallback to the configured version.
    return parent::getVersion();
  }

  /**
   * {@inheritdoc}
   */
  public function validateLibrary(): array {
    $messages = parent::validateLibrary();

    // Check if the library is present.
    $path = $this->getLibraryPath();
    if (!$path) {
      $messages[] = $this->t('Tick library not found in /libraries folder.');
      return $messages;
    }

    // Check for required files.
    $required_files = [
      'dist/core/tick.core.global.min.js',
      'dist/core/tick.core.min.css',
    ];

    foreach ($required_files as $file) {
      if (!file_exists(DRUPAL_ROOT . '/' . $path . '/' . $file)) {
        $messages[] = $this->t('Required file @file not found.', ['@file' => $file]);
      }
    }

    // Check for extension files if any extensions are enabled.
    $enabled_extensions = $this->getEnabledExtensions();
    foreach ($enabled_extensions as $ext_id) {
      if (!isset(self::EXTENSIONS[$ext_id])) {
        continue;
      }

      $extension = self::EXTENSIONS[$ext_id];
      $found = FALSE;

      // Check if at least one extension file exists.
      foreach (['files', 'js'] as $key) {
        if (isset($extension[$key])) {
          foreach ($extension[$key] as $file) {
            if (file_exists(DRUPAL_ROOT . '/' . $path . '/' . $file)) {
              $found = TRUE;
              break 2;
            }
          }
        }
      }

      if (!$found) {
        $messages[] = $this->t('Tick extension @ext not found in library.', [
          '@ext' => $extension['label'],
        ]);
      }
    }

    return $messages;
  }

}
