<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\countdown\Service\CountdownLibraryDiscoveryInterface;
use Drupal\countdown\Utility\CdnUrlBuilder;
use Drupal\countdown\Utility\ConfigAccessor;
use Drupal\countdown\Utility\LibraryPathResolver;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Countdown Library plugins.
 *
 * @see \Drupal\countdown\Annotation\CountdownLibrary
 * @see \Drupal\countdown\Plugin\CountdownLibraryPluginInterface
 * @see \Drupal\countdown\CountdownLibraryPluginManager
 * @see plugin_api
 */
abstract class CountdownLibraryPluginBase extends PluginBase implements CountdownLibraryPluginInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The countdown library discovery service.
   *
   * @var \Drupal\countdown\Service\CountdownLibraryDiscoveryInterface
   */
  protected CountdownLibraryDiscoveryInterface $libraryDiscovery;

  /**
   * The library path resolver.
   *
   * @var \Drupal\countdown\Utility\LibraryPathResolver
   */
  protected LibraryPathResolver $pathResolver;

  /**
   * The CDN URL builder.
   *
   * @var \Drupal\countdown\Utility\CdnUrlBuilder
   */
  protected CdnUrlBuilder $cdnBuilder;

  /**
   * The configuration accessor.
   *
   * @var \Drupal\countdown\Utility\ConfigAccessor
   */
  protected ConfigAccessor $configAccessor;

  /**
   * Cached library path.
   *
   * @var string|null|false
   */
  protected string|null|false $libraryPath = NULL;

  /**
   * Cached installed version.
   *
   * @var string|null|false
   */
  protected string|null|false $installedVersion = NULL;

  /**
   * Constructs a CountdownLibraryPluginBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\countdown\Service\CountdownLibraryDiscoveryInterface $library_discovery
   *   The countdown library discovery service.
   * @param \Drupal\countdown\Utility\LibraryPathResolver $path_resolver
   *   The library path resolver.
   * @param \Drupal\countdown\Utility\CdnUrlBuilder $cdn_builder
   *   The CDN URL builder.
   * @param \Drupal\countdown\Utility\ConfigAccessor $config_accessor
   *   The configuration accessor.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ModuleHandlerInterface $module_handler,
    FileSystemInterface $file_system,
    LoggerInterface $logger,
    CacheBackendInterface $cache,
    CountdownLibraryDiscoveryInterface $library_discovery,
    LibraryPathResolver $path_resolver,
    CdnUrlBuilder $cdn_builder,
    ConfigAccessor $config_accessor,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleHandler = $module_handler;
    $this->fileSystem = $file_system;
    $this->logger = $logger;
    $this->cache = $cache;
    $this->libraryDiscovery = $library_discovery;
    $this->pathResolver = $path_resolver;
    $this->cdnBuilder = $cdn_builder;
    $this->configAccessor = $config_accessor;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    // Create config accessor.
    $config_accessor = new ConfigAccessor($container->get('config.factory'));

    // Create path resolver with debug mode from config.
    $path_resolver = new LibraryPathResolver(
      $container->get('file_system'),
      $container->get('logger.channel.countdown'),
      $config_accessor->isDebugMode()
    );

    // Create CDN builder.
    $cdn_builder = new CdnUrlBuilder();

    // Return the fully constructed plugin instance.
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
      $container->get('file_system'),
      $container->get('logger.channel.countdown'),
      $container->get('cache.discovery'),
      $container->get('countdown.library_discovery'),
      $path_resolver,
      $cdn_builder,
      $config_accessor
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): string {
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return (string) $this->pluginDefinition['description'];
  }

  /**
   * {@inheritdoc}
   */
  public function getType(): string {
    return $this->pluginDefinition['type'] ?? 'external';
  }

  /**
   * {@inheritdoc}
   */
  public function isInstalled(): bool {
    return $this->getLibraryPath() !== NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getLibraryPath(): ?string {
    if ($this->libraryPath === NULL) {
      $this->libraryPath = $this->findLibrary() ?: FALSE;
    }

    return $this->libraryPath !== FALSE ? $this->libraryPath : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateInstallation(string $path): bool {
    return $this->pathResolver->validateInstallation(
      $path,
      $this->getRequiredFiles(),
      $this->getAlternativePaths()
    );
  }

  /**
   * Finds the library installation path.
   *
   * @return string|null
   *   The library path or NULL if not found.
   */
  protected function findLibrary(): ?string {
    $path = $this->pathResolver->findLibrary(
      $this->getPluginId(),
      $this->getPossibleFolderNames()
    );

    if ($path && $this->validateInstallation(ltrim($path, '/'))) {
      return $path;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function detectVersion(string $path): ?string {
    $path = ltrim($path, '/');
    $base_path = DRUPAL_ROOT . '/' . $path;

    // Try multiple detection strategies.
    $strategies = [
      'detectVersionFromPackageJson',
      'detectVersionFromComposerJson',
      'detectVersionFromBowerJson',
      'detectVersionFromVersionFiles',
    ];

    foreach ($strategies as $method) {
      if (method_exists($this, $method)) {
        $version = $this->$method($base_path);
        if ($version) {
          if ($this->configAccessor->isDebugMode()) {
            $this->logger->debug('Detected version @version for library @library using @method', [
              '@version' => $version,
              '@library' => $this->getPluginId(),
              '@method' => $method,
            ]);
          }
          return $version;
        }
      }
    }

    // Allow plugins to implement custom detection.
    $version = $this->detectVersionCustom($base_path);
    if ($version) {
      return $version;
    }

    if ($this->configAccessor->isDebugMode()) {
      $this->logger->debug('Could not detect version for library @library at path @path', [
        '@library' => $this->getPluginId(),
        '@path' => $path,
      ]);
    }

    return NULL;
  }

  /**
   * Custom version detection for specific libraries.
   *
   * @param string $base_path
   *   The base path of the library.
   *
   * @return string|null
   *   The detected version or NULL.
   */
  protected function detectVersionCustom(string $base_path): ?string {
    // Override in child classes for custom detection.
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredVersion(): ?string {
    return $this->pluginDefinition['version'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getInstalledVersion(): ?string {
    if ($this->installedVersion === NULL) {
      $path = $this->getLibraryPath();
      if ($path) {
        $this->installedVersion = $this->detectVersion(ltrim($path, '/')) ?: FALSE;
      }
      else {
        $this->installedVersion = FALSE;
      }
    }

    return $this->installedVersion !== FALSE ? $this->installedVersion : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function versionMeetsRequirements(): bool {
    $installed = $this->getInstalledVersion();
    $required = $this->getRequiredVersion();

    if (!$installed) {
      return FALSE;
    }

    if (!$required) {
      return TRUE;
    }

    return version_compare($installed, $required, '>=');
  }

  /**
   * {@inheritdoc}
   */
  public function getAssets(bool $minified = TRUE): array {
    // Delegate to asset map if defined.
    $asset_map = $this->getAssetMap();
    if (!empty($asset_map)) {
      return $this->buildAssetsFromMap($asset_map, $minified);
    }

    // Legacy path-based asset discovery.
    $path = $this->getLibraryPath();
    if (!$path) {
      return [];
    }

    $files = $this->pluginDefinition['files'] ?? [];
    $variant = $minified ? 'production' : 'development';
    $assets = [];

    foreach (['js', 'css'] as $type) {
      if (isset($files[$type][$variant])) {
        $file_path = $files[$type][$variant];
        $full_path = $path . '/' . $file_path;

        if (file_exists(DRUPAL_ROOT . '/' . $full_path)) {
          $assets[$type][] = [
            'path' => $full_path,
            'external' => FALSE,
            'minified' => $minified,
            'preprocess' => $minified,
          ];
        }
      }
    }

    return $assets;
  }

  /**
   * Gets the asset map for this library.
   *
   * @return array
   *   Asset map with 'local' and 'cdn' keys.
   */
  protected function getAssetMap(): array {
    // Override in child classes to provide declarative asset mapping.
    return [];
  }

  /**
   * Builds assets from an asset map.
   *
   * @param array $asset_map
   *   The asset map.
   * @param bool $minified
   *   Whether to use minified assets.
   *
   * @return array
   *   Array of assets keyed by type.
   */
  protected function buildAssetsFromMap(array $asset_map, bool $minified): array {
    $method = $this->configAccessor->getLoadingMethod();
    $assets = [];

    if ($method === 'cdn') {
      // Build CDN assets.
      $provider = $this->configAccessor->getCdnProvider();
      if (isset($asset_map['cdn'][$provider])) {
        $cdn_config = $asset_map['cdn'][$provider];

        // Add JS.
        if (isset($cdn_config['js'])) {
          $assets['js'][] = [
            'path' => $cdn_config['js'],
            'external' => TRUE,
            'minified' => TRUE,
            'preprocess' => FALSE,
            'attributes' => ['crossorigin' => 'anonymous'],
          ];
        }

        // Add CSS.
        if (isset($cdn_config['css'])) {
          $assets['css'][] = [
            'path' => $cdn_config['css'],
            'external' => TRUE,
            'minified' => TRUE,
            'preprocess' => FALSE,
          ];
        }
      }
    }
    else {
      // Build local assets.
      $path = $this->getLibraryPath();
      if ($path && isset($asset_map['local'])) {
        $variant = $minified ? 'production' : 'development';

        foreach (['js', 'css'] as $type) {
          if (isset($asset_map['local'][$type][$variant])) {
            $file_path = $path . '/' . $asset_map['local'][$type][$variant];
            if (file_exists(DRUPAL_ROOT . '/' . $file_path)) {
              $assets[$type][] = [
                'path' => $file_path,
                'external' => FALSE,
                'minified' => $minified,
                'preprocess' => $minified,
              ];
            }
          }
        }
      }
    }

    return $assets;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequiredFiles(): array {
    return $this->pluginDefinition['required_files'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getAlternativePaths(): array {
    return $this->pluginDefinition['alternative_paths'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getPossibleFolderNames(): array {
    $names = $this->pluginDefinition['folder_names'] ?? [];

    // Always include the plugin ID.
    array_unshift($names, $this->getPluginId());

    // Add NPM package name if different.
    $npm_package = $this->getNpmPackage();
    if ($npm_package && !in_array($npm_package, $names)) {
      $names[] = $npm_package;
    }

    return array_unique($names);
  }

  /**
   * {@inheritdoc}
   */
  public function getInitFunction(): ?string {
    return $this->pluginDefinition['init_function'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getNpmPackage(): ?string {
    return $this->pluginDefinition['npm_package'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getCdnConfig(): ?array {
    // Priority 1: Check annotation-based CDN configuration.
    if (isset($this->pluginDefinition['cdn']) && is_array($this->pluginDefinition['cdn']) && !empty($this->pluginDefinition['cdn'])) {
      return $this->pluginDefinition['cdn'];
    }

    // Priority 2: Fall back to asset map CDN configuration.
    $asset_map = $this->getAssetMap();
    if (isset($asset_map['cdn']) && is_array($asset_map['cdn']) && !empty($asset_map['cdn'])) {
      return $asset_map['cdn'];
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getDependencies(): array {
    return $this->pluginDefinition['dependencies'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getHomepage(): ?string {
    return $this->pluginDefinition['homepage'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getRepository(): ?string {
    return $this->pluginDefinition['repository'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthor(): ?string {
    return $this->pluginDefinition['author'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getLicense(): ?string {
    return $this->pluginDefinition['license'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): array {
    if (!$this->isInstalled()) {
      return [
        'installed' => FALSE,
        'version_status' => 'not_installed',
        'messages' => [$this->t('Library is not installed.')],
        'severity' => 'error',
      ];
    }

    $status = [
      'installed' => TRUE,
      'version_status' => 'unknown',
      'messages' => [],
      'severity' => 'info',
    ];

    $installed_version = $this->getInstalledVersion();
    $required_version = $this->getRequiredVersion();

    if ($installed_version) {
      $status['messages'][] = $this->t('Installed version: @version', [
        '@version' => $installed_version,
      ]);

      if ($required_version) {
        if ($this->versionMeetsRequirements()) {
          $status['version_status'] = 'ok';
          $status['messages'][] = $this->t('Version meets requirements (minimum: @min)', [
            '@min' => $required_version,
          ]);
        }
        else {
          $status['version_status'] = 'outdated';
          $status['severity'] = 'warning';
          $status['messages'][] = $this->t('Version is outdated. Minimum required: @min', [
            '@min' => $required_version,
          ]);
        }
      }
      else {
        $status['version_status'] = 'ok';
        $status['messages'][] = $this->t('No specific version requirements.');
      }
    }
    else {
      $status['messages'][] = $this->t('Version could not be detected.');
      $status['severity'] = 'warning';
    }

    return $status;
  }

  /**
   * {@inheritdoc}
   */
  public function buildLibraryDefinition(bool $minified = TRUE): array {
    $definition = [
      'version' => $this->getInstalledVersion() ?: '1.0',
      'dependencies' => $this->getDependencies(),
    ];

    $assets = $this->getAssets($minified);

    // Add JS assets.
    if (!empty($assets['js'])) {
      foreach ($assets['js'] as $js) {
        $definition['js'][$js['path']] = [
          'minified' => $js['minified'],
          'preprocess' => $js['preprocess'],
          'attributes' => ['defer' => TRUE],
        ];
      }
    }

    // Add CSS assets.
    if (!empty($assets['css'])) {
      foreach ($assets['css'] as $css) {
        $definition['css']['theme'][$css['path']] = [
          'minified' => $css['minified'],
          'preprocess' => $css['preprocess'],
        ];
      }
    }

    // Add drupalSettings if needed.
    $settings = $this->getLibrarySettings();
    if (!empty($settings)) {
      $definition['drupalSettings']['countdown'][$this->getPluginId()] = $settings;
    }

    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function getLibrarySettings(): array {
    return [
      'library' => $this->getPluginId(),
      'initFunction' => $this->getInitFunction(),
      'version' => $this->getInstalledVersion(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return $this->pluginDefinition['weight'] ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function isExperimental(): bool {
    return $this->pluginDefinition['experimental'] ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function resetCache(): void {
    $this->libraryPath = NULL;
    $this->installedVersion = NULL;
    $this->pathResolver->clearCache();
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtensions(): bool {
    return !empty($this->getAvailableExtensions());
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableExtensions(): array {
    // Default implementation - override in child classes if needed.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getExtensionGroups(): array {
    // Default implementation - override in child classes if needed.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildAttachments(array $config): array {
    $attachments = [];

    // Determine the library to attach based on type and method.
    if ($this->getType() === 'core') {
      // Core library uses timer definitions.
      $library_name = $config['variant'] ? 'countdown/timer.min' : 'countdown/timer';
      $attachments['#attached']['library'][] = $library_name;

      // Also attach the core integration.
      $integration_name = $config['variant'] ? 'countdown/integration.core.min' : 'countdown/integration.core';
      $attachments['#attached']['library'][] = $integration_name;
    }
    else {
      // External libraries: use countdown.libraries.yml static definitions.
      $plugin_id = $this->getPluginId();

      // Check is installed Tick library first.
      $has_tick = $this->libraryDiscovery->isInstalled('tick');

      $library_name = $has_tick && $plugin_id === 'flip' ? 'tick' : $plugin_id;

      if ($config['method'] === 'cdn') {
        // CDN libraries have _cdn suffix.
        $library_name .= '_cdn';
        if ($config['variant']) {
          $library_name .= '.min';
        }
      }
      else {
        // Local libraries.
        if ($config['variant']) {
          $library_name .= '.min';
        }
      }

      $attachments['#attached']['library'][] = 'countdown/' . $library_name;

      // Attach the specific integration file.
      $integration_name = $config['variant']
        ? 'countdown/integration.' . $plugin_id . '.min'
        : 'countdown/integration.' . $plugin_id;
      $attachments['#attached']['library'][] = $integration_name;
    }



    // Create a web-accessible URL for the integration base path.
    // Use the Drupal base path to handle subdirectory installations.
    $base_url = \Drupal::request()->getBasePath();

    // Get the module path relative to the Drupal root.
    $module_path = $base_url . '/' . $this->moduleHandler->getModule('countdown')->getPath();
    $integration_path = $module_path . '/js/integrations';

    // Add drupalSettings for JavaScript initialization.
    $attachments['#attached']['drupalSettings']['countdown'] = [
      'activeLibrary' => $this->getPluginId(),
      'libraryType' => $this->getType(),
      'loadingMethod' => $config['method'],
      'initFunction' => $this->getInitFunction(),
      'settings' => $this->getLibrarySettings(),
      'modulePath' => $module_path,
      'integrationBasePath' => $integration_path,
    ];

    // Always attach the base integration library.
    $base_integration = $config['variant'] ? 'countdown/integration.min' : 'countdown/integration';
    $attachments['#attached']['library'][] = $base_integration;

    // Add RTL support if enabled.
    if (!empty($config['rtl'])) {
      $attachments['#attached']['library'][] = 'core/drupal.rtl';
    }

    return $attachments;
  }

  /**
   * {@inheritdoc}
   */
  public function validateExtensions(array $extensions): array {
    // Default implementation for plugins without extensions.
    if (!$this->hasExtensions()) {
      return [];
    }

    // Override in child classes that have extensions.
    return [];
  }

  /**
   * Detects version from package.json file.
   *
   * @param string $base_path
   *   The base path of the library.
   *
   * @return string|null
   *   The detected version or NULL.
   */
  protected function detectVersionFromPackageJson(string $base_path): ?string {
    $locations = ['/package.json', '/dist/package.json', '/src/package.json'];

    foreach ($locations as $location) {
      $file = $base_path . $location;
      if (file_exists($file)) {
        try {
          $content = file_get_contents($file);
          $data = json_decode($content, TRUE);

          if (json_last_error() === JSON_ERROR_NONE && !empty($data['version'])) {
            return $this->normalizeVersion($data['version']);
          }
        }
        catch (\Exception $e) {
          // Continue to next location.
        }
      }
    }

    return NULL;
  }

  /**
   * Detects version from composer.json file.
   *
   * @param string $base_path
   *   The base path of the library.
   *
   * @return string|null
   *   The detected version or NULL.
   */
  protected function detectVersionFromComposerJson(string $base_path): ?string {
    $file = $base_path . '/composer.json';

    if (file_exists($file)) {
      try {
        $content = file_get_contents($file);
        $data = json_decode($content, TRUE);

        if (json_last_error() === JSON_ERROR_NONE && !empty($data['version'])) {
          return $this->normalizeVersion($data['version']);
        }
      }
      catch (\Exception $e) {
        // Continue.
      }
    }

    return NULL;
  }

  /**
   * Detects version from bower.json file.
   *
   * @param string $base_path
   *   The base path of the library.
   *
   * @return string|null
   *   The detected version or NULL.
   */
  protected function detectVersionFromBowerJson(string $base_path): ?string {
    $file = $base_path . '/bower.json';

    if (file_exists($file)) {
      try {
        $content = file_get_contents($file);
        $data = json_decode($content, TRUE);

        if (json_last_error() === JSON_ERROR_NONE && !empty($data['version'])) {
          return $this->normalizeVersion($data['version']);
        }
      }
      catch (\Exception $e) {
        // Continue.
      }
    }

    return NULL;
  }

  /**
   * Detects version from version files.
   *
   * @param string $base_path
   *   The base path of the library.
   *
   * @return string|null
   *   The detected version or NULL.
   */
  protected function detectVersionFromVersionFiles(string $base_path): ?string {
    $files = ['VERSION', 'VERSION.txt', 'version.txt', '.version', 'version'];

    foreach ($files as $file) {
      $version_file = $base_path . '/' . $file;

      if (file_exists($version_file)) {
        $content = trim(file_get_contents($version_file));

        if (preg_match('/^v?(\d+\.\d+(?:\.\d+)?(?:[-+].+)?)$/i', $content, $matches)) {
          return $this->normalizeVersion($matches[1]);
        }
      }
    }

    return NULL;
  }

  /**
   * Normalizes version string.
   *
   * @param string $version
   *   The raw version string.
   *
   * @return string
   *   The normalized version string.
   */
  protected function normalizeVersion(string $version): string {
    $version = preg_replace('/^v/i', '', trim($version));
    $version = trim($version, '"\'');
    return $version;
  }

}
