<?php

declare(strict_types=1);

namespace Drupal\countdown\Service;

/**
 * Interface for countdown library discovery service using Plugin System.
 *
 * @package Drupal\countdown\Service
 */
interface CountdownLibraryDiscoveryInterface {

  /**
   * Discover all available countdown library plugins.
   *
   * @param bool $reset
   *   Whether to reset the cache and force re-discovery.
   *
   * @return array
   *   Array of discovered libraries keyed by plugin ID.
   */
  public function discover(bool $reset = FALSE): array;

  /**
   * Get information about a specific library plugin.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return array|null
   *   Library information array or NULL if not found.
   */
  public function getLibrary(string $library_id): ?array;

  /**
   * Get all installed library plugins.
   *
   * @return array
   *   Array of installed libraries keyed by plugin ID.
   */
  public function getAvailableLibraries(): array;

  /**
   * Check if a library plugin is installed.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return bool
   *   TRUE if the library is installed.
   */
  public function isInstalled(string $library_id): bool;

  /**
   * Get the path to a library.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return string|null
   *   The library path or NULL if not installed.
   */
  public function getLibraryPath(string $library_id): ?string;

  /**
   * Get library assets for inclusion.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return array
   *   Array of assets keyed by type (js, css).
   */
  public function getLibraryAssets(string $library_id): array;

  /**
   * Get the installed version of a library.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return string|null
   *   The installed version or NULL.
   */
  public function getInstalledVersion(string $library_id): ?string;

  /**
   * Check if library version meets requirements.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return bool
   *   TRUE if version meets requirements.
   */
  public function versionMeetsRequirements(string $library_id): bool;

  /**
   * Get library status information.
   *
   * @param string $library_id
   *   The plugin ID.
   *
   * @return array
   *   Status information array.
   */
  public function getLibraryStatus(string $library_id): array;

  /**
   * Clear the discovery cache.
   */
  public function clearCache(): void;

}
