<?php

/**
 * @file
 * API documentation for the Countdown module.
 */

/**
 * @addtogroup hooks
 * @{
 */

/**
 * Alter countdown library plugin definitions.
 *
 * This hook allows modules to alter the definitions of countdown library
 * plugins after they have been discovered by the plugin manager.
 *
 * @param array $definitions
 *   An array of plugin definitions, keyed by plugin ID.
 *
 * @see \Drupal\countdown\CountdownLibraryPluginManager
 * @see \Drupal\countdown\Annotation\CountdownLibrary
 */
function hook_countdown_library_info_alter(array &$definitions) {
  // Change the label of the FlipClock library.
  if (isset($definitions['flipclock'])) {
    $definitions['flipclock']['label'] = t('Custom FlipClock Label');
  }

  // Add a custom CDN configuration for a library.
  if (isset($definitions['flipdown'])) {
    $definitions['flipdown']['cdn']['custom'] = [
      'css' => '//my-cdn.com/flipdown.css',
      'js' => '//my-cdn.com/flipdown.js',
    ];
  }

  // Mark a library as experimental.
  if (isset($definitions['tick'])) {
    $definitions['tick']['experimental'] = TRUE;
  }
}

/**
 * Alter the library definition before it's registered.
 *
 * This hook allows modules to alter the Drupal library definition
 * that will be used for including the countdown library assets.
 *
 * @param array $definition
 *   The library definition array that will be registered.
 * @param string $library_id
 *   The ID of the active library.
 *
 * @see \Drupal\countdown\Service\CountdownLibraryManager::getLibraryDefinition()
 */
function hook_countdown_library_definition_alter(array &$definition, string $library_id) {
  // Add custom JavaScript settings.
  $definition['drupalSettings']['countdown']['custom'] = [
    'option1' => 'value1',
    'option2' => 'value2',
  ];

  // Add additional dependencies.
  $definition['dependencies'][] = 'mymodule/custom-library';

  // Modify asset attributes.
  if (isset($definition['js'])) {
    foreach ($definition['js'] as &$options) {
      $options['attributes']['async'] = TRUE;
    }
  }
}

/**
 * @} End of "addtogroup hooks".
 */

/**
 * @defgroup countdown_plugins Countdown Library Plugins
 * @{
 * Define custom countdown library plugins.
 *
 * Countdown library plugins allow developers to add support for additional
 * countdown/timer JavaScript libraries. Each plugin is a class that extends
 * \Drupal\countdown\Plugin\CountdownLibraryPluginBase and is annotated with
 * \Drupal\countdown\Annotation\CountdownLibrary.
 *
 * Plugins are discovered automatically from the Plugin/CountdownLibrary
 * namespace of any enabled module.
 *
 * @section sec_creating Creating a Countdown Library Plugin
 *
 * To create a custom countdown library plugin:
 *
 * 1. Create a PHP class in your module's src/Plugin/CountdownLibrary directory.
 * 2. Extend \Drupal\countdown\Plugin\CountdownLibraryPluginBase.
 * 3. Add the @CountdownLibrary annotation with required properties.
 * 4. Implement any methods needed to customize behavior.
 *
 * @section sec_example Example Plugin
 *
 * @code
 * <?php
 *
 * namespace Drupal\mymodule\Plugin\CountdownLibrary;
 *
 * use Drupal\countdown\Plugin\CountdownLibraryPluginBase;
 *
 * /**
 *  * Custom countdown library plugin.
 *  *
 *  * @CountdownLibrary(
 *  *   id = "my_timer",
 *  *   label = @Translation("My Custom Timer"),
 *  *   description = @Translation("A custom countdown timer library"),
 *  *   type = "external",
 *  *   homepage = "https://example.com/my-timer",
 *  *   version = "1.0.0",
 *  *   npm_package = "my-timer",
 *  *   folder_names = {
 *  *     "my-timer",
 *  *     "MyTimer"
 *  *   },
 *  *   files = {
 *  *     "css" = {
 *  *       "development" = "dist/timer.css",
 *  *       "production" = "dist/timer.min.css"
 *  *     },
 *  *     "js" = {
 *  *       "development" = "dist/timer.js",
 *  *       "production" = "dist/timer.min.js"
 *  *     }
 *  *   },
 *  *   required_files = {
 *  *     "dist/timer.js",
 *  *     "dist/timer.css"
 *  *   },
 *  *   init_function = "MyTimer",
 *  *   author = "My Company",
 *  *   license = "MIT",
 *  *   cdn = {
 *  *     "jsdelivr" = {
 *  *       "css" = "//cdn.jsdelivr.net/npm/my-timer@1.0.0/dist/timer.min.css",
 *  *       "js" = "//cdn.jsdelivr.net/npm/my-timer@1.0.0/dist/timer.min.js"
 *  *     }
 *  *   }
 *  * )
 *  * /
 * class MyTimer extends CountdownLibraryPluginBase {
 *
 *   /**
 *    * {@inheritdoc}
 *    * /
 *   protected function detectVersionCustom(string $base_path): ?string {
 *     // Custom version detection logic.
 *     $version_file = $base_path . '/VERSION';
 *     if (file_exists($version_file)) {
 *       return trim(file_get_contents($version_file));
 *     }
 *     return NULL;
 *   }
 *
 *   /**
 *    * {@inheritdoc}
 *    * /
 *   protected function getLibrarySettings(): array {
 *     $settings = parent::getLibrarySettings();
 *
 *     // Add library-specific settings.
 *     $settings['theme'] = 'default';
 *     $settings['format'] = 'HH:mm:ss';
 *
 *     return $settings;
 *   }
 *
 * }
 * @endcode
 *
 * @section sec_annotation Annotation Properties
 *
 * The @CountdownLibrary annotation supports the following properties:
 *
 * - **id** (required): The unique machine name of the library.
 * - **label** (required): The human-readable name of the library.
 * - **description** (required): A brief description of the library.
 * - **type**: Either 'core' or 'external' (default: 'external').
 * - **homepage**: The library's homepage URL.
 * - **repository**: The library's source code repository URL.
 * - **version**: The minimum required version.
 * - **npm_package**: The NPM package name if available.
 * - **folder_names**: Array of possible folder names to search for.
 * - **files**: Array defining CSS and JS file paths for development/production.
 * - **required_files**: Array of files that must exist for validation.
 * - **alternative_paths**: Array of alternative file path sets.
 * - **init_function**: The JavaScript initialization function name.
 * - **author**: The library author or organization.
 * - **license**: The library license.
 * - **dependencies**: Array of Drupal library dependencies.
 * - **cdn**: CDN configuration for different providers.
 * - **weight**: Plugin weight for sorting (default: 0).
 * - **experimental**: Whether the library is experimental (default: FALSE).
 * - **api_version**: API version compatibility (default: '1.0').
 *
 * @section sec_methods Key Methods to Override
 *
 * When creating a custom plugin, you may want to override these methods:
 *
 * - **detectVersionCustom()**: Implement custom version detection logic.
 * - **getLibrarySettings()**: Provide library-specific drupalSettings.
 * - **validateInstallation()**: Custom validation logic for the library.
 * - **buildLibraryDefinition()**: Customize the Drupal library definition.
 * - **getAssets()**: Customize how assets are discovered and returned.
 *
 * @section sec_services Using Plugin Manager Services
 *
 * The plugin manager service provides several useful methods:
 *
 * @code
 * // Get the plugin manager service.
 * $plugin_manager = \Drupal::service('plugin.manager.countdown_library');
 *
 * // Get all plugins.
 * $all_plugins = $plugin_manager->getAllPlugins();
 *
 * // Get only installed plugins.
 * $installed_plugins = $plugin_manager->getInstalledPlugins();
 *
 * // Get a specific plugin.
 * $plugin = $plugin_manager->getPlugin('my_timer');
 *
 * // Get plugin options for form selects.
 * $options = $plugin_manager->getPluginOptions(TRUE, TRUE, FALSE);
 *
 * // Validate a plugin.
 * $messages = $plugin_manager->validatePlugin('my_timer');
 *
 * // Get CDN-capable plugins.
 * $cdn_plugins = $plugin_manager->getCdnCapablePlugins();
 * @endcode
 *
 * @}
 */

/**
 * @defgroup countdown_examples Example Implementations
 * @{
 * Example implementations for common use cases.
 *
 * @section sec_auto_load Auto-loading a Specific Library
 *
 * To automatically load a specific countdown library on certain pages:
 *
 * @code
 * /**
 *  * Implements hook_page_attachments().
 *  * /
 * function mymodule_page_attachments(array &$attachments) {
 *   $current_path = \Drupal::service('path.current')->getPath();
 *
 *   // Load FlipClock on specific pages.
 *   if (strpos($current_path, '/special-countdown') === 0) {
 *     $attachments['#attached']['library'][] = 'countdown/countdown.flipclock';
 *   }
 * }
 * @endcode
 *
 * @section sec_render_array Using in Render Arrays
 *
 * To use a countdown library in a render array:
 *
 * @code
 * $build['countdown'] = [
 *   '#theme' => 'countdown',
 *   '#library' => 'flipclock',
 *   '#target_date' => '2025-12-31 23:59:59',
 *   '#format' => 'DailyCounter',
 *   '#attributes' => [
 *     'id' => 'my-countdown',
 *     'class' => ['special-countdown'],
 *   ],
 *   '#settings' => [
 *     'countdown' => TRUE,
 *     'showSeconds' => TRUE,
 *   ],
 *   '#attached' => [
 *     'library' => ['countdown/countdown.flipclock'],
 *   ],
 * ];
 * @endcode
 *
 * @section sec_js_init JavaScript Initialization
 *
 * To initialize a countdown timer via JavaScript:
 *
 * @code
 * (function (Drupal, drupalSettings) {
 *   'use strict';
 *
 *   Drupal.behaviors.myCountdown = {
 *     attach: function (context, settings) {
 *       const element = document.getElementById('my-countdown');
 *
 *       if (element && !element.classList.contains('countdown-initialized')) {
 *         // Get the library settings.
 *         const librarySettings = settings.countdown.flipclock;
 *
 *         // Initialize based on library type.
 *         if (window.FlipClock) {
 *           const clock = new FlipClock(element, {
 *             countdown: true,
 *             clockFace: 'DailyCounter'
 *           });
 *         }
 *
 *         element.classList.add('countdown-initialized');
 *       }
 *     }
 *   };
 * })(Drupal, drupalSettings);
 * @endcode
 *
 * @}
 */
