/**
 * @file
 * Admin form enhancements for countdown blocks.
 *
 * Provides preview updates and dependent field handling for the block
 * configuration form.
 */

(function ($, Drupal, drupalSettings) {
  'use strict';

  /**
   * Admin form behavior for countdown blocks.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countdownBlockAdmin = {
    attach: function (context, settings) {
      // Handle preview updates when form values change.
      const $form = $('form.block-form', context).once('countdown-block-admin');

      if (!$form.length) {
        return;
      }

      // Find preview container if it exists.
      const $preview = $('#countdown-block-preview', context);

      if ($preview.length) {
        // Update preview when relevant fields change.
        const updatePreview = function() {
          // Get current form values.
          const library = $('[name="settings[library_wrapper][library]"]', $form).val();
          const targetDate = $('[name="settings[target_date][date]"]', $form).val();
          const targetTime = $('[name="settings[target_date][time]"]', $form).val();

          if (!library || !targetDate) {
            return;
          }

          // Clean up existing preview instance.
          if (Drupal.countdown && Drupal.countdown.stop) {
            $preview.find('.countdown-timer').each(function() {
              Drupal.countdown.stop(this);
            });
          }

          // Clear preview content.
          $preview.html('<div class="countdown-preview-loading">' +
            Drupal.t('Loading preview...') + '</div>');

          // Build preview element.
          const previewElement = document.createElement('div');
          previewElement.className = 'countdown-timer countdown-block-preview';
          previewElement.dataset.countdownLibrary = library;
          previewElement.dataset.countdownTarget = targetDate + 'T' + (targetTime || '00:00:00');

          // Get library-specific settings.
          const librarySettings = {};
          $('[name^="settings[library_settings_wrapper][library_settings][' + library + ']"]', $form).each(function() {
            const name = this.name.replace(/^.*\[([^\]]+)\]$/, '$1');
            librarySettings[name] = $(this).val();
          });

          previewElement.dataset.countdownSettings = JSON.stringify({
            library: library,
            librarySettings: librarySettings,
            finishMessage: $('[name="settings[finish_message]"]', $form).val() || "Time's up!"
          });

          // Replace loading message with preview element.
          $preview.html(previewElement);

          // Initialize the preview using the main integration.
          if (Drupal.countdown && Drupal.countdown.loadIntegration) {
            // Ensure the library is set in drupalSettings.
            if (!drupalSettings.countdown) {
              drupalSettings.countdown = {};
            }
            drupalSettings.countdown.activeLibrary = library;

            // Load and initialize.
            Drupal.countdown.loadIntegration(library, function() {
              Drupal.countdown.initialize(previewElement.parentNode, drupalSettings);
            });
          }
        };

        // Debounced preview update.
        let updateTimeout;
        const debouncedUpdate = function() {
          clearTimeout(updateTimeout);
          updateTimeout = setTimeout(updatePreview, 500);
        };

        // Watch for changes to relevant fields.
        $form.on('change', '[name^="settings[library"]', debouncedUpdate);
        $form.on('change', '[name^="settings[target_date]"]', debouncedUpdate);
        $form.on('change', '[name^="settings[library_settings_wrapper]"]', debouncedUpdate);
        $form.on('change', '[name="settings[finish_message]"]', debouncedUpdate);
      }

      // Handle method-dependent library options.
      const $methodRadios = $('[name="settings[method]"]', $form);
      const $libraryWrapper = $('#countdown-library-wrapper', $form);

      $methodRadios.on('change', function() {
        const method = $(this).val();

        // Update CDN provider visibility.
        const $cdnProvider = $('[name="settings[library_wrapper][cdn_provider]"]', $libraryWrapper);
        if (method === 'cdn') {
          $cdnProvider.closest('.form-item').show();
        } else {
          $cdnProvider.closest('.form-item').hide();
        }
      });

      // Trigger initial state.
      $methodRadios.filter(':checked').trigger('change');
    }
  };

})(jQuery, Drupal, drupalSettings);
