/**
 * @file
 * Bridge integration for countdown blocks.
 *
 * This script handles block-specific initialization by preparing the DOM
 * and delegating to the main countdown integration system.
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Countdown block bridge behavior.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countdownBlockBridge = {
    attach: function (context, settings) {
      // Find all countdown blocks that haven't been initialized yet.
      const blocks = once('countdown-block-bridge', '.countdown-block', context);

      if (!blocks.length) {
        return;
      }

      // Process each block.
      blocks.forEach(function(element) {
        const blockId = element.dataset.blockId;
        const blockLibrary = element.dataset.countdownLibrary;

        // Find the inner countdown element.
        const innerTimer = element.querySelector('.countdown');

        if (!innerTimer) {
          console.warn('CountdownBlock: No inner timer element found for block', blockId);
          return;
        }

        // Skip if already initialized by main integration.
        if (innerTimer.classList.contains('countdown-initialized')) {
          return;
        }

        // Get block configuration.
        const blockConfig = settings.countdown;

        if (!blockConfig) {
          console.warn('CountdownBlock: No configuration found for block', blockId);
          return;
        }

        // Set data attributes from block configuration. These will be read
        // by the main integration system.
        innerTimer.dataset.countdownLibrary = blockLibrary || blockConfig.library;
        innerTimer.dataset.countdownTarget = element.dataset.countdownTarget || blockConfig.target;
        innerTimer.dataset.countdownSettings = element.dataset.countdownSettings || JSON.stringify(blockConfig.settings);
        innerTimer.dataset.blockId = blockId;

        // The main integration will handle initialization. We just ensure
        // the element is properly marked for processing.
        if (!innerTimer.hasAttribute('data-once')) {
          // Mark the element as ready for main integration processing.
          innerTimer.removeAttribute('data-once');
        }

        // If the main integration is already loaded, trigger it directly
        // for this specific element with the block's library.
        if (Drupal.countdown && Drupal.countdown.initializeCountdown) {
          // Store the current global active library to restore later.
          const globalActiveLibrary = settings.countdown?.activeLibrary;

          // Temporarily set the block's library as active.
          if (settings.countdown) {
            settings.countdown.activeLibrary = blockConfig.library;
          }

          // Initialize this specific countdown.
          Drupal.countdown.initializeCountdown(innerTimer, settings);

          // Restore the global active library.
          if (settings.countdown && globalActiveLibrary !== undefined) {
            settings.countdown.activeLibrary = globalActiveLibrary;
          }
        }
      });
    },

    detach: function (context, settings, trigger) {
      // Handle cleanup when blocks are removed.
      if (trigger === 'unload') {
        const blocks = context.querySelectorAll('.countdown-block');

        blocks.forEach(function(element) {
          const innerTimer = element.querySelector('.countdown');

          if (innerTimer && Drupal.countdown && Drupal.countdown.stop) {
            // Stop the countdown instance.
            Drupal.countdown.stop(innerTimer);
          }
        });
      }
    }
  };

})(Drupal, drupalSettings, once);
