/**
 * @file
 * Bridge integration for countdown fields.
 *
 * This script handles field-specific initialization by preparing the DOM
 * and delegating to the main countdown integration system.
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Countdown field bridge behavior.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countdownFieldBridge = {
    attach: function (context, settings) {
      // Find all countdown fields that haven't been initialized yet.
      const fields = once('countdown-field-bridge', '.countdown-field', context);

      if (!fields.length) {
        return;
      }

      // Process each field.
      fields.forEach(function (element) {
        const fieldId = element.dataset.countdownFieldId;
        const delta = element.dataset.countdownDelta;
        const fullId = fieldId + '-' + delta;

        // Find the inner countdown element.
        const innerTimer = element.querySelector('.countdown-timer');

        if (!innerTimer) {
          console.warn('CountdownField: No inner timer element found for field', fullId);
          return;
        }

        // Skip if already initialized by main integration.
        if (innerTimer.classList.contains('countdown-initialized')) {
          return;
        }

        // Get field configuration from drupalSettings.
        const fieldConfig = settings.countdownField?.fields?.[fullId];

        if (!fieldConfig) {
          console.warn('CountdownField: No configuration found for field', fullId);
          return;
        }

        // Set data attributes from field configuration. These will be read
        // by the main integration system.
        innerTimer.dataset.countdownLibrary = fieldConfig.library;
        innerTimer.dataset.countdownTarget = fieldConfig.target;
        innerTimer.dataset.countdownSettings = JSON.stringify(fieldConfig.settings);
        innerTimer.dataset.fieldId = fieldId;
        innerTimer.dataset.fieldDelta = delta;

        // Add the correct library-specific class for styling.
        if (fieldConfig.library) {
          innerTimer.classList.add('countdown-' + fieldConfig.library);
        }

        // The main integration will handle initialization. We just ensure
        // the element is properly marked for processing.
        if (!innerTimer.hasAttribute('data-once')) {
          // Mark the element as ready for main integration processing.
          innerTimer.removeAttribute('data-once');
        }

        // If the main integration is already loaded, trigger it directly
        // for this specific element.
        if (Drupal.countdown && Drupal.countdown.initializeCountdown) {
          Drupal.countdown.initializeCountdown(innerTimer, settings);
        }
      });
    },

    detach: function (context, settings, trigger) {
      // Handle cleanup when fields are removed.
      if (trigger === 'unload') {
        const fields = context.querySelectorAll('.countdown-field');

        fields.forEach(function (element) {
          const innerTimer = element.querySelector('.countdown-timer');

          if (innerTimer && Drupal.countdown && Drupal.countdown.stop) {
            // Stop the countdown instance.
            Drupal.countdown.stop(innerTimer);
          }
        });
      }
    }
  };

})(Drupal, drupalSettings, once);
