<?php

declare(strict_types=1);

namespace Drupal\countdown\Plugin;

use Drupal\Component\Plugin\DerivativeInspectionInterface;
use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines an interface for Countdown Library plugins.
 *
 * @see \Drupal\countdown\Annotation\CountdownLibrary
 * @see \Drupal\countdown\CountdownLibraryPluginManager
 * @see \Drupal\countdown\Plugin\CountdownLibraryPluginBase
 * @see plugin_api
 */
interface CountdownLibraryPluginInterface extends PluginInspectionInterface, DerivativeInspectionInterface {

  /**
   * Returns the library label.
   *
   * @return string
   *   The library label.
   */
  public function getLabel(): string;

  /**
   * Returns the library description.
   *
   * @return string
   *   The library description.
   */
  public function getDescription(): string;

  /**
   * Returns the library type (core or external).
   *
   * @return string
   *   Either 'core' or 'external'.
   */
  public function getType(): string;

  /**
   * Checks if the library is installed.
   *
   * @return bool
   *   TRUE if the library is installed, FALSE otherwise.
   */
  public function isInstalled(): bool;

  /**
   * Gets the installed library path.
   *
   * @return string|null
   *   The library path relative to DRUPAL_ROOT, or NULL if not installed.
   */
  public function getLibraryPath(): ?string;

  /**
   * Validates the library installation at the given path.
   *
   * @param string $path
   *   The path to validate (relative to DRUPAL_ROOT).
   *
   * @return bool
   *   TRUE if the library is valid at this path, FALSE otherwise.
   */
  public function validateInstallation(string $path): bool;

  /**
   * Detects the installed version of the library.
   *
   * @param string $path
   *   The library path (relative to DRUPAL_ROOT).
   *
   * @return string|null
   *   The detected version string, or NULL if not detected.
   */
  public function detectVersion(string $path): ?string;

  /**
   * Gets the minimum required version.
   *
   * @return string|null
   *   The minimum version required, or NULL if any version is acceptable.
   */
  public function getRequiredVersion(): ?string;

  /**
   * Gets the installed version.
   *
   * @return string|null
   *   The installed version, or NULL if not detected.
   */
  public function getInstalledVersion(): ?string;

  /**
   * Checks if the installed version meets requirements.
   *
   * @return bool
   *   TRUE if version requirements are met, FALSE otherwise.
   */
  public function versionMeetsRequirements(): bool;

  /**
   * Gets the library assets for inclusion.
   *
   * @param bool $minified
   *   Whether to return minified assets.
   *
   * @return array
   *   Array of assets keyed by type (js, css).
   */
  public function getAssets(bool $minified = TRUE): array;

  /**
   * Gets the required files for validation.
   *
   * @return array
   *   Array of required file paths relative to library root.
   */
  public function getRequiredFiles(): array;

  /**
   * Gets alternative file paths.
   *
   * @return array
   *   Array of alternative path sets.
   */
  public function getAlternativePaths(): array;

  /**
   * Gets possible folder names for discovery.
   *
   * @return array
   *   Array of possible folder names to search for.
   */
  public function getPossibleFolderNames(): array;

  /**
   * Gets the initialization function name.
   *
   * @return string|null
   *   The JavaScript initialization function name.
   */
  public function getInitFunction(): ?string;

  /**
   * Gets the NPM package name.
   *
   * @return string|null
   *   The NPM package name if available.
   */
  public function getNpmPackage(): ?string;

  /**
   * Gets CDN configuration.
   *
   * @return array|null
   *   CDN configuration array or NULL if not available.
   */
  public function getCdnConfig(): ?array;

  /**
   * Gets the library dependencies.
   *
   * @return array
   *   Array of Drupal library dependencies.
   */
  public function getDependencies(): array;

  /**
   * Gets the library homepage URL.
   *
   * @return string|null
   *   The homepage URL or NULL.
   */
  public function getHomepage(): ?string;

  /**
   * Gets the library repository URL.
   *
   * @return string|null
   *   The repository URL or NULL.
   */
  public function getRepository(): ?string;

  /**
   * Gets the library author.
   *
   * @return string|null
   *   The author name or NULL.
   */
  public function getAuthor(): ?string;

  /**
   * Gets the library license.
   *
   * @return string|null
   *   The license identifier or NULL.
   */
  public function getLicense(): ?string;

  /**
   * Gets the library status information.
   *
   * @return array
   *   Status array containing:
   *   - installed: Boolean installation status.
   *   - version_status: Version status ('ok', 'outdated', 'unknown').
   *   - messages: Array of status messages.
   *   - severity: Severity level ('info', 'warning', 'error').
   */
  public function getStatus(): array;

  /**
   * Builds the library definition for hook_library_info_build().
   *
   * @param bool $minified
   *   Whether to use minified assets.
   *
   * @return array
   *   Library definition array compatible with *.libraries.yml structure.
   */
  public function buildLibraryDefinition(bool $minified = TRUE): array;

  /**
   * Gets the plugin weight for sorting.
   *
   * @return int
   *   The plugin weight.
   */
  public function getWeight(): int;

  /**
   * Checks if the library is experimental.
   *
   * @return bool
   *   TRUE if experimental, FALSE otherwise.
   */
  public function isExperimental(): bool;

  /**
   * Resets any cached data for this library.
   */
  public function resetCache(): void;

  /**
   * Checks if this library has extensions.
   *
   * @return bool
   *   TRUE if library has extensions, FALSE otherwise.
   */
  public function hasExtensions(): bool;

  /**
   * Gets available extensions for this library.
   *
   * Extensions are additional components that can be loaded with the library
   * (e.g., themes, views, fonts for Tick library).
   *
   * @return array
   *   Array of extension information with keys:
   *   - label: Human-readable name.
   *   - description: Extension description.
   *   - files: Array of file paths.
   *   - cdn: CDN configuration per provider.
   */
  public function getAvailableExtensions(): array;

  /**
   * Gets extension groups for organizational purposes.
   *
   * Groups help organize extensions in the UI (e.g., Views, Themes, Fonts).
   *
   * @return array
   *   Array of groups with keys:
   *   - label: Group label.
   *   - description: Group description.
   *   - extensions: Array of extension IDs in this group.
   *   - weight: Sort weight.
   */
  public function getExtensionGroups(): array;

  /**
   * Gets library-specific settings for drupalSettings.
   *
   * @return array
   *   Settings array to be added to drupalSettings.countdown.
   */
  public function getLibrarySettings(): array;

  /**
   * Validates that required extensions are available.
   *
   * This method checks if the specified extensions are properly installed
   * and available for use with the library.
   *
   * @param array $extensions
   *   Array of extension IDs to validate.
   *
   * @return array
   *   Validation error messages array; empty if all extensions are valid.
   */
  public function validateExtensions(array $extensions): array;

  /**
   * Builds page attachments for this library.
   *
   * This method creates attachments (libraries, drupalSettings, etc.)
   * for adding the library to a page.
   *
   * @param array $config
   *   Configuration array containing:
   *   - method: Loading method ('local' or 'cdn').
   *   - variant: Whether to use minified assets.
   *   - cdn_provider: The CDN provider to use.
   *   - rtl: Whether RTL support is enabled.
   *   - debug: Whether debug mode is enabled.
   *
   * @return array
   *   Attachments array suitable for page attachments.
   */
  public function buildAttachments(array $config): array;

  /**
   * Builds configuration form elements for this library.
   *
   * This method creates form elements specific to this library's
   * configuration options. It is used by blocks, fields, and other
   * components that need library-specific settings.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $default_values
   *   Default configuration values.
   */
  public function buildConfigurationForm(array &$form, FormStateInterface $form_state, array $default_values = []): void;

  /**
   * Validates the configuration form for this library.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void;

  /**
   * Gets default configuration values for this library.
   *
   * @return array
   *   Default configuration array with library-specific settings.
   */
  public function getDefaultConfiguration(): array;

  /**
   * Transforms configuration values for JavaScript consumption.
   *
   * This method converts form configuration values into the format
   * expected by the JavaScript library initialization code.
   *
   * @param array $configuration
   *   The configuration values from the form.
   *
   * @return array
   *   Settings array formatted for JavaScript.
   */
  public function getJavaScriptSettings(array $configuration): array;

}
