<?php

declare(strict_types=1);

namespace Drupal\countdown\Service;

/**
 * Interface for countdown library manager service using Plugin System.
 *
 * This interface defines all methods required for managing countdown libraries
 * including local and CDN loading, validation, configuration, and caching.
 *
 * @package Drupal\countdown\Service
 */
interface CountdownLibraryManagerInterface {

  /**
   * Get the active library plugin ID.
   *
   * @return string
   *   The active library plugin ID.
   */
  public function getActiveLibrary(): string;

  /**
   * Set the active library plugin.
   *
   * @param string $library_id
   *   The plugin ID to set as active.
   *
   * @return bool
   *   TRUE on success.
   *
   * @throws \InvalidArgumentException
   *   If the library plugin is missing or unsuitable for the current method.
   */
  public function setActiveLibrary(string $library_id): bool;

  /**
   * Get the current loading method.
   *
   * @return string
   *   Either 'local' or 'cdn'.
   */
  public function getLoadingMethod(): string;

  /**
   * Set the loading method.
   *
   * @param string $method
   *   Either 'local' or 'cdn'.
   *
   * @return bool
   *   TRUE on success.
   *
   * @throws \InvalidArgumentException
   *   If the method is invalid.
   */
  public function setLoadingMethod(string $method): bool;

  /**
   * Get the current CDN provider.
   *
   * @return string
   *   The CDN provider name (e.g., 'jsdelivr', 'cdnjs', 'unpkg', 'custom').
   */
  public function getCdnProvider(): string;

  /**
   * Set the CDN provider.
   *
   * @param string $provider
   *   The CDN provider name.
   *
   * @return bool
   *   TRUE on success.
   *
   * @throws \InvalidArgumentException
   *   If the provider is invalid.
   */
  public function setCdnProvider(string $provider): bool;

  /**
   * Get all available CDN providers.
   *
   * @return array
   *   Array of available CDN provider names.
   */
  public function getAvailableCdnProviders(): array;

  /**
   * Get CDN providers that support a specific library.
   *
   * @param string $library_id
   *   The library plugin ID.
   *
   * @return array
   *   Array of CDN provider names that support this library.
   */
  public function getCdnProvidersForLibrary(string $library_id): array;

  /**
   * Get all CDN providers across all libraries.
   *
   * This method scans all CDN-capable plugins and returns a unique list
   * of all available CDN providers.
   *
   * @return array
   *   Array of all CDN provider names keyed by provider ID.
   */
  public function getCdnProviders(): array;

  /**
   * Get the library definition for hook_library_info_build().
   *
   * @return array
   *   The library definition array compatible with *.libraries.yml structure.
   */
  public function getLibraryDefinition(): array;

  /**
   * Build CDN library definition.
   *
   * @param \Drupal\countdown\Plugin\CountdownLibraryPluginInterface $plugin
   *   The library plugin.
   * @param bool $minified
   *   Whether to use minified assets.
   *
   * @return array
   *   The CDN library definition.
   */
  public function buildCdnDefinition($plugin, bool $minified = TRUE): array;

  /**
   * Get available library options for forms.
   *
   * @param string|null $for_method
   *   Optional method to filter options ('local' or 'cdn').
   *   If NULL, uses current method.
   *
   * @return array
   *   Array of options keyed by plugin ID.
   */
  public function getAvailableLibraryOptions(string $for_method = NULL): array;

  /**
   * Validate the current library configuration.
   *
   * @return array
   *   Array of validation messages (errors and warnings).
   */
  public function validateLibraryConfiguration(): array;

  /**
   * Get library requirements for hook_requirements().
   *
   * @return array
   *   Array of requirements compatible with hook_requirements().
   */
  public function getLibraryRequirements(): array;

  /**
   * Check if auto-loading is enabled.
   *
   * @return bool
   *   TRUE if auto-loading is enabled.
   */
  public function isAutoLoadEnabled(): bool;

  /**
   * Set auto-loading state.
   *
   * @param bool $enabled
   *   Whether to enable auto-loading.
   *
   * @return bool
   *   TRUE on success.
   */
  public function setAutoLoad(bool $enabled): bool;

  /**
   * Get the build variant (minified/development).
   *
   * @return bool
   *   TRUE for minified/production, FALSE for development.
   */
  public function getBuildVariant(): bool;

  /**
   * Set the build variant.
   *
   * @param bool $minified
   *   TRUE for minified/production, FALSE for development.
   *
   * @return bool
   *   TRUE on success.
   */
  public function setBuildVariant(bool $minified): bool;

  /**
   * Get enabled asset types.
   *
   * @return array
   *   Array with 'js' and 'css' keys and boolean values.
   */
  public function getAssetTypes(): array;

  /**
   * Set enabled asset types.
   *
   * @param array $types
   *   Array with 'js' and/or 'css' keys and boolean values.
   *
   * @return bool
   *   TRUE on success.
   */
  public function setAssetTypes(array $types): bool;

  /**
   * Switch to a different library and optionally change method.
   *
   * @param string $library_id
   *   The library plugin ID.
   * @param string|null $method
   *   Optional loading method ('local' or 'cdn').
   *
   * @return bool
   *   TRUE on success.
   *
   * @throws \InvalidArgumentException
   *   If the library or method is invalid.
   */
  public function switchLibrary(string $library_id, string $method = NULL): bool;

  /**
   * Get detailed information about a library.
   *
   * @param string $library_id
   *   The library plugin ID.
   *
   * @return array|null
   *   Array of library information or NULL if not found.
   */
  public function getLibraryInfo(string $library_id): ?array;

  /**
   * Get information about all libraries.
   *
   * @return array
   *   Array of library information keyed by plugin ID.
   */
  public function getAllLibrariesInfo(): array;

  /**
   * Check if a library is compatible with a loading method.
   *
   * @param string $library_id
   *   The library plugin ID.
   * @param string $method
   *   The loading method ('local' or 'cdn').
   *
   * @return bool
   *   TRUE if the library is compatible with the method.
   */
  public function isLibraryCompatible(string $library_id, string $method): bool;

  /**
   * Clear all caches related to countdown libraries.
   */
  public function clearCache(): void;

}
