/**
 * @file
 * Integration for the FlipDown.js library.
 *
 * Handles initialization of FlipDown countdown instances with configuration
 * from drupalSettings and element data attributes.
 */

(function (Drupal) {
  'use strict';

  /**
   * Initialize a FlipDown timer.
   *
   * @param {Element} element
   *   The DOM element to initialize as a timer.
   * @param {Object} settings
   *   The settings object from drupalSettings.
   */
  function initializeFlipDown(element, settings) {
    // Validate library availability.
    if (typeof FlipDown === 'undefined') {
      Drupal.countdown.utils.handleError(element, 'FlipDown library not loaded', 'flipdown');
      return;
    }

    // Resolve settings using shared utility.
    const config = Drupal.countdown.utils.resolveCountdownSettings(element, settings, 'flipdown');

    // Get target date.
    const targetDate = config.target_date || element.dataset.countdownTarget;
    if (!targetDate) {
      Drupal.countdown.utils.handleError(element, 'No target date specified', 'flipdown');
      return;
    }

    // Convert to Unix timestamp in seconds.
    const timestamp = Math.floor(new Date(targetDate).getTime() / 1000);
    const now = Math.floor(Date.now() / 1000);

    // Check if already expired.
    if (timestamp <= now) {
      Drupal.countdown.utils.showExpiredMessage(element, config, 'flipdown');
      return;
    }

    // Create inner container for FlipDown
    const innerContainer = document.createElement('div');
    innerContainer.id = 'flipdown-' + Math.random().toString(36).substr(2, 9);

    // Add theme class
    const theme = config.theme || 'dark';
    innerContainer.classList.add('flipdown');
    innerContainer.classList.add('flipdown__theme-' + theme);

    // Add custom class if specified
    if (config.custom_class) {
      innerContainer.classList.add(config.custom_class);
    }

    // Append to countdown display area
    const displayArea = element.querySelector('.countdown-display');
    if (displayArea) {
      // Clear any loading message
      displayArea.innerHTML = '';
      displayArea.appendChild(innerContainer);
    } else {
      // Fallback if display area doesn't exist
      element.appendChild(innerContainer);
    }

    // Build FlipDown configuration
    const flipdownConfig = {};

    // Set headings if specified
    if (config.headings && Array.isArray(config.headings)) {
      flipdownConfig.headings = config.headings;
    }

    try {
      // Create and start FlipDown
      const flipdown = new FlipDown(timestamp, innerContainer.id, flipdownConfig);
      flipdown.start();

      // Handle completion
      flipdown.ifEnded(function () {
        Drupal.countdown.utils.showExpiredMessage(element, config, 'flipdown');
      });

      // Store instance for cleanup
      Drupal.countdown.storeInstance(element, {
        instance: flipdown,
        innerContainer: innerContainer,
        stop: function () {
          if (flipdown.countdown) {
            clearInterval(flipdown.countdown);
          }
          // Remove the inner container
          if (innerContainer.parentNode) {
            innerContainer.parentNode.removeChild(innerContainer);
          }
        }
      });

      // Mark as initialized
      element.classList.add('countdown-initialized');

      // Debug output if enabled
      if (config.debug_mode || config.debug) {
        console.log('FlipDown initialized:', {
          element: element,
          timestamp: timestamp,
          config: flipdownConfig,
          settings: config
        });
      }

      // Dispatch initialization event
      Drupal.countdown.utils.dispatchEvent(element, 'initialized', {
        library: 'flipdown',
        element: element,
        settings: config,
        instance: flipdown
      });
    }
    catch (error) {
      console.error('Countdown: Failed to create FlipDown', error);
      Drupal.countdown.utils.handleError(element, 'Failed to create FlipDown: ' + error.message, 'flipdown');
    }
  }

  // Register loader with countdown system
  Drupal.countdown.registerLoader('flipdown', initializeFlipDown);

})(Drupal);
