# Countdown

A highly configurable Drupal module that provides countdown and count-up
blocks with both server-side and real-time JavaScript rendering options.

## 📑 TABLE OF CONTENTS

- [Introduction](#introduction)
- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Configuration](#configuration)
- [Maintainers](#maintainers)

## INTRODUCTION

The Countdown module adds powerful countdown/count-up timer blocks to your
Drupal site. Whether you need a simple static countdown that updates on page
refresh or a dynamic real-time timer with millisecond precision, this module
provides the flexibility and features you need.

Perfect for:
- Event countdowns (conferences, product launches, deadlines)
- Time-based content reveals
- Milestone tracking (days since launch, time until deadline)
- Limited-time offers and promotions
- Anniversary and celebration timers

**Project page:** https://www.drupal.org/project/countdown
**Issue queue:** https://www.drupal.org/project/issues/countdown

## FEATURES

### Core Functionality
- **Dual Render Modes**
  - Server-side (static): SEO-friendly, lightweight, no JavaScript required
  - Real-time JavaScript: Dynamic updates with configurable precision

### Display Options
- **Multiple Display Formats**
  - Default digital clock (HH:MM:SS)
  - Verbose with labels (3 days, 2 hours, 1 minute)
  - Compact notation (3d 2h 1m)
  - Custom templates with flexible tokens

- **Time Precision Levels**
  - Minutes (HH:MM)
  - Seconds (HH:MM:SS)
  - Tenths of a second (HH:MM:SS.m)
  - Hundredths (HH:MM:SS.mm)
  - Milliseconds (HH:MM:SS.mmm)

### Advanced Features
- **Timer Modes**
  - Count down to future events
  - Count up from past events
  - Automatic elapsed time after completion

- **Completion Actions**
  - Hide the countdown block
  - Display custom message
  - Redirect to URL
  - Reload the page
  - Switch to elapsed time mode
  - Trigger custom JavaScript events

- **Technical Capabilities**
  - Timezone support with optional display
  - Start offset for fine-tuning
  - Drift compensation for long-running timers
  - JavaScript API for programmatic control
  - Debug mode for development
  - Event triggers for custom integrations

### JavaScript API
Full programmatic control through the `Drupal.countdown` namespace:
- Start, stop, pause, resume, reset timers
- Listen to timer events (tick, complete, elapsed)
- Get timer instances and state
- Control multiple timers independently

### Performance Optimized
- Lightweight static mode for high-traffic sites
- Efficient drift compensation algorithm
- Configurable update frequency
- Minimal DOM manipulation
- No external dependencies for core functionality

## REQUIREMENTS

This module requires:
- Drupal 8.8 or higher (including Drupal 9, 10, and 11)
- PHP 7.3 or higher
- Block module (Drupal core)

No external libraries or dependencies are required. The module includes its
own optimized CountdownTimer JavaScript library.

## INSTALLATION

### Using Composer (Recommended)
```bash
  composer require drupal/countdown
  drush en countdown
```

### Manual Installation

1. Download the module from https://www.drupal.org/project/countdown
2. Extract to `/modules/contrib/countdown`
3. Navigate to Extend (`/admin/modules`)
4. Enable the Countdown module
5. Clear caches

## CONFIGURATION

### Quick Start

1. Navigate to Structure > Block layout (`/admin/structure/block`)
2. Click Place block in your desired region
3. Search for "Countdown" and click Place block
4. Configure your countdown:
  - Enter an event name
  - Set the target date/time
  - Choose display settings
  - Save the block

### Basic Settings

#### Event Configuration
- **Event Name**: The name of your event (required)
- **Event URL**: Optional link to more information
- **Target Date/Time**: The countdown target using intuitive date/time fields

#### Display Mode
- **Server-side**: Updates on page refresh, SEO-friendly, works without JS
- **Real-time**: Dynamic countdown with configurable precision and formats

### Advanced Configuration

#### Real-time Mode Options
- **Timer Mode**: Count down or count up
- **Precision**: Choose update frequency (minutes to milliseconds)
- **Display Style**: Auto, verbose, compact, or custom template
- **Custom Templates**: Create your own format using tokens:
  - `DD` / `D`: Days (padded/unpadded)
  - `HH` / `H`: Hours (padded/unpadded)
  - `MM` / `M`: Minutes (padded/unpadded)
  - `SS` / `S`: Seconds (padded/unpadded)
  - `mmm`: Milliseconds
  - `mm`: Hundredths
  - `m`: Tenths

#### Completion Actions
Configure what happens when countdown reaches zero:
- **None**: Stop at 00:00:00
- **Hide**: Fade out the block
- **Message**: Show custom text
- **Redirect**: Navigate to URL
- **Reload**: Refresh the page
- **Elapsed**: Switch to count-up mode
- **Event**: Trigger custom JavaScript

#### Technical Settings
- **Start Offset**: Adjust countdown by seconds
- **Auto-start**: Begin immediately on page load
- **Drift Compensation**: Maintain accuracy for long timers
- **Timezone**: Set event timezone with optional display
- **JavaScript Events**: Enable for custom integrations
- **Debug Mode**: Console logging for development

### JavaScript API Usage

#### Get Timer Instance

    // Get timer by block ID
    var timer = Drupal.countdown.getTimer("block_id");

    // Get all active timers
    var allTimers = Drupal.countdown.getAllTimers();

#### Control Timer

    // Timer operations
    Drupal.countdown.controlTimer("block_id", "start");
    Drupal.countdown.controlTimer("block_id", "pause");
    Drupal.countdown.controlTimer("block_id", "resume");
    Drupal.countdown.controlTimer("block_id", "stop");
    Drupal.countdown.controlTimer("block_id", "reset");

#### Listen to Events

    // Timer tick event
    $("#countdown-block_id").on("countdown:tick", function(e, time, timer) {
      console.log("Time:", time.days, "days", time.hours, "hours");
    });

    // Completion event
    $("#countdown-block_id").on("countdown:complete", function(e, time, timer) {
      // Custom completion logic
    });

    // Available events:
    // countdown:start, countdown:pause, countdown:resume
    // countdown:stop, countdown:tick, countdown:complete
    // countdown:elapsed, countdown:custom-complete

#### Custom Integration Example

    (function ($, Drupal) {
      Drupal.behaviors.myCountdownIntegration = {
        attach: function (context, settings) {
          // React to countdown completion
          $('.countdown-timer', context).once('my-block').each(function() {
            $(this).on('countdown:complete', function(e, time, timer) {
              // Show a modal, update content, trigger animation, etc.
              alert('Event has started!');
            });
          });
        }
      };
    })(jQuery, Drupal);

### Custom Template Examples

#### Event Landing Page

    "Only DD days, HH hours, and MM minutes until launch!"

#### Precise Scientific Timer

    "HH:MM:SS.mmm"

#### Compact Mobile Display

    "DDd HHh MMm"

### Performance Tips

1. **Choose the Right Mode**
  - Use static mode for SEO-critical pages
  - Use real-time for interactive experiences

2. **Optimize Precision**
  - Use seconds for most countdowns
  - Reserve milliseconds for special effects

3. **Multiple Timers**
  - Each timer runs independently
  - Consider static mode for many timers on one page

4. **Caching**
  - Static mode respects page cache
  - Real-time mode automatically bypasses cache

### Troubleshooting

#### Countdown Shows Wrong Time
- Check timezone settings in block configuration
- Verify server time is correct
- Consider time offset setting

#### Real-time Mode Not Working
- Ensure JavaScript is enabled
- Check browser console for errors
- Enable debug mode for diagnostics
- Clear Drupal and browser caches

#### Completion Actions Not Triggering
- Verify countdown mode (not count-up)
- Check completion action is configured
- For custom events, ensure listener is attached

#### Performance Issues
- Reduce precision (use seconds instead of milliseconds)
- Enable drift compensation
- Consider static mode for high-traffic pages
- Check for JavaScript conflicts

## MAINTAINERS

### Current Maintainers
- **David Norman** (deekayen) - https://www.drupal.org/u/deekayen
- **Joachim Noreiko** (joachim) - https://www.drupal.org/u/joachim
- **Jacob Neher** (bocaj) - https://www.drupal.org/u/bocaj
- **Using Session** (usingsession) - https://www.drupal.org/u/usingsession
- **Rahul Nahar** (rahul.nahar001) - https://www.drupal.org/u/rahulnahar001
- **Mahyar SBT** (mahyarsbt) - https://www.drupal.org/u/mahyarsbt

### Credits
This module was originally created by the Drupal community and has been
enhanced with modern JavaScript capabilities, real-time rendering, and
extensive customization options.

### Support
- **Documentation**: https://www.drupal.org/docs/contributed-modules/countdown
- **Issue Queue**: https://www.drupal.org/project/issues/countdown
- **Security Issues**: Follow standard Drupal security reporting procedures

### Contributing
We welcome contributions! Please see the issue queue for current tasks or
submit patches for review. All contributions must follow Drupal coding
standards.
