/**
 * @file
 * Admin form enhancements for the Countdown module.
 *
 * Provides interactive improvements for date/time field selection.
 */

(function ($, Drupal) {
  'use strict';

  /**
   * Enhance countdown admin date/time fields.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.countdownAdminEnhance = {
    attach: function (context, settings) {
      // Initialize date/time field enhancements.
      $('.countdown-datetime-fieldset', context).once('countdown-admin-enhance').each(function () {
        var $fieldset = $(this);

        // Auto-update max days when month/year changes.
        setupDayLimitUpdater($fieldset);

        // Add quick date presets.
        addDatePresets($fieldset);

        // Enhance number inputs with increment/decrement buttons.
        enhanceNumberInputs($fieldset);

        // Add validation feedback.
        setupValidationFeedback($fieldset);
      });
    }
  };

  /**
   * Update day field max value based on selected month/year.
   *
   * @param {jQuery} $fieldset
   *   The fieldset containing the date/time fields.
   */
  function setupDayLimitUpdater($fieldset) {
    var $month = $fieldset.find('.countdown-month-select');
    var $year = $fieldset.find('.countdown-year-input');
    var $day = $fieldset.find('.countdown-day-input');

    // Function to update the max day value.
    function updateDayMax() {
      var month = parseInt($month.val(), 10);
      var year = parseInt($year.val(), 10);

      if (month && year) {
        // Calculate days in month.
        var daysInMonth = new Date(year, month, 0).getDate();
        $day.attr('max', daysInMonth);

        // Adjust current value if it exceeds the new max.
        if (parseInt($day.val(), 10) > daysInMonth) {
          $day.val(daysInMonth);
        }
      }
    }

    // Bind to change events.
    $month.on('change', updateDayMax);
    $year.on('change', updateDayMax);

    // Initial update.
    updateDayMax();
  }

  /**
   * Add quick preset buttons for common date selections.
   *
   * @param {jQuery} $fieldset
   *   The fieldset containing the date/time fields.
   */
  function addDatePresets($fieldset) {
    // Create preset button container.
    var $presets = $('<div class="countdown-date-presets"></div>');

    // Define preset options.
    var presets = [
      {label: Drupal.t('Tomorrow'), days: 1},
      {label: Drupal.t('Next Week'), days: 7},
      {label: Drupal.t('Next Month'), months: 1},
      {label: Drupal.t('Next Year'), years: 1}
    ];

    // Create buttons for each preset.
    presets.forEach(function(preset) {
      var $button = $('<button type="button" class="button button--small">' + preset.label + '</button>');

      $button.on('click', function(e) {
        e.preventDefault();
        setDateFromPreset($fieldset, preset);
      });

      $presets.append($button);
    });

    // Insert presets after the description.
    $fieldset.find('.fieldset__description').after($presets);
  }

  /**
   * Set date fields based on preset values.
   *
   * @param {jQuery} $fieldset
   *   The fieldset containing the date/time fields.
   * @param {Object} preset
   *   The preset configuration object.
   */
  function setDateFromPreset($fieldset, preset) {
    var now = new Date();
    var target = new Date(now);

    // Apply preset offset.
    if (preset.days) {
      target.setDate(target.getDate() + preset.days);
    }
    if (preset.months) {
      target.setMonth(target.getMonth() + preset.months);
    }
    if (preset.years) {
      target.setFullYear(target.getFullYear() + preset.years);
    }

    // Update form fields.
    $fieldset.find('.countdown-month-select').val(target.getMonth() + 1).trigger('change');
    $fieldset.find('.countdown-day-input').val(target.getDate());
    $fieldset.find('.countdown-year-input').val(target.getFullYear()).trigger('change');

    // Keep current time.
    $fieldset.find('.countdown-hour-input').val(now.getHours());
    $fieldset.find('.countdown-minute-input').val(now.getMinutes());
    $fieldset.find('.countdown-second-input').val(0);

    // Add visual feedback.
    $fieldset.find('.countdown-datetime-item').addClass('is-valid');
    setTimeout(function() {
      $fieldset.find('.countdown-datetime-item').removeClass('is-valid');
    }, 2000);
  }

  /**
   * Enhance number inputs with increment/decrement functionality.
   *
   * @param {jQuery} $fieldset
   *   The fieldset containing the date/time fields.
   */
  function enhanceNumberInputs($fieldset) {
    $fieldset.find('input[type="number"]').each(function() {
      var $input = $(this);

      // Skip if already enhanced.
      if ($input.hasClass('enhanced')) {
        return;
      }

      // Add keyboard shortcuts.
      $input.on('keydown', function(e) {
        var step = e.shiftKey ? 10 : 1;

        if (e.key === 'ArrowUp') {
          e.preventDefault();
          incrementValue($input, step);
        }
        else if (e.key === 'ArrowDown') {
          e.preventDefault();
          incrementValue($input, -step);
        }
      });

      // Add mousewheel support.
      $input.on('wheel', function(e) {
        if ($input.is(':focus')) {
          e.preventDefault();
          var delta = e.originalEvent.deltaY < 0 ? 1 : -1;
          incrementValue($input, delta);
        }
      });

      $input.addClass('enhanced');
    });
  }

  /**
   * Increment or decrement input value within bounds.
   *
   * @param {jQuery} $input
   *   The input element to modify.
   * @param {number} delta
   *   The amount to change the value by.
   */
  function incrementValue($input, delta) {
    var current = parseInt($input.val(), 10) || 0;
    var min = parseInt($input.attr('min'), 10);
    var max = parseInt($input.attr('max'), 10);

    var newValue = current + delta;

    // Apply bounds.
    if (!isNaN(min) && newValue < min) {
      newValue = min;
    }
    if (!isNaN(max) && newValue > max) {
      newValue = max;
    }

    $input.val(newValue).trigger('change');
  }

  /**
   * Setup real-time validation feedback.
   *
   * @param {jQuery} $fieldset
   *   The fieldset containing the date/time fields.
   */
  function setupValidationFeedback($fieldset) {
    var $inputs = $fieldset.find('input[type="number"], select');

    $inputs.on('change blur', function() {
      var $input = $(this);
      var $item = $input.closest('.countdown-datetime-item');

      // Basic validation.
      if ($input.is(':invalid') || !$input.val()) {
        $item.removeClass('is-valid').addClass('is-invalid');
      }
      else {
        $item.removeClass('is-invalid').addClass('is-valid');

        // Remove valid class after a delay.
        setTimeout(function() {
          $item.removeClass('is-valid');
        }, 1500);
      }
    });
  }

})(jQuery, Drupal);
