/**
 * @file
 * Integration layer between Drupal and CountdownTimer library.
 *
 * Provides Drupal behaviors and utilities for managing countdown timers,
 * handling completion actions, and exposing a programmatic API.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  // Use WeakMap for better memory management of timer instances.
  const timerInstances = new WeakMap();

  /**
   * Initialize countdown blocks on the page.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attach countdown functionality.
   * @prop {Drupal~behaviorDetach} detach
   *   Detach countdown functionality.
   */
  Drupal.behaviors.countdownIntegration = {
    attach: function (context, settings) {
      // Process each countdown container that hasn't been initialized.
      const containers = context.querySelectorAll('.countdown-timer-container:not([data-countdown-initialized])');

      containers.forEach(function (container) {
        const blockId = container.dataset.blockId;
        const configKey = 'countdown_' + blockId;

        // Validate configuration exists.
        if (!drupalSettings.countdown || !drupalSettings.countdown[configKey]) {
          console.warn('Countdown: No configuration found for block', blockId);
          return;
        }

        const config = drupalSettings.countdown[configKey];

        // Skip static mode blocks.
        if (config.render_mode === 'static') {
          return;
        }

        // Find the display element within the container.
        const displayElement = container.querySelector('#countdown-display-' + blockId);
        if (!displayElement) {
          console.warn('Countdown: Display element not found for block', blockId);
          return;
        }

        // Initialize the timer targeting the display element.
        initializeTimer(container, displayElement, config, blockId);
      });
    },

    detach: function (context, settings, trigger) {
      // Clean up timers when elements are removed.
      if (trigger === 'unload') {
        const containers = context.querySelectorAll('.countdown-timer-container[data-countdown-initialized]');

        containers.forEach(function (container) {
          const timer = timerInstances.get(container);
          if (timer) {
            timer.destroy();
            timerInstances.delete(container);
          }
        });
      }
    }
  };

  /**
   * Initialize a countdown timer for a specific element.
   *
   * @param {HTMLElement} container
   *   The container element for the countdown.
   * @param {HTMLElement} displayElement
   *   The display element where countdown will render.
   * @param {Object} config
   *   Configuration from Drupal settings.
   * @param {string} blockId
   *   The block identifier.
   */
  function initializeTimer(container, displayElement, config, blockId) {
    // Check if CountdownTimer library is loaded.
    if (typeof CountdownTimer === 'undefined') {
      console.error('Countdown: CountdownTimer library not loaded');
      return;
    }

    try {
      // Build timer configuration targeting the display element.
      const timerConfig = buildTimerConfig(displayElement, config);

      // Create timer instance.
      const timer = new CountdownTimer(timerConfig);

      // For elapsed mode from start, manually adjust the start time.
      if (config.elapsed_from_start === true && config.offset) {
        timer.state.startTime = Date.now() - parseInt(config.offset, 10);
      }

      // Store instance reference on container for management.
      timerInstances.set(container, timer);

      // Store original config for elapsed mode switching.
      container.dataset.originalConfig = JSON.stringify(config);
      container.dataset.originalTimestamp = config.timestamp;

      // Remove placeholder if it exists.
      const placeholder = displayElement.querySelector('[data-placeholder="true"]');
      if (placeholder) {
        placeholder.remove();
      }

      // Mark container as initialized.
      container.setAttribute('data-countdown-initialized', 'true');

      // Handle timezone display if configured.
      if (config.show_timezone && config.timezone) {
        addTimezoneDisplay(container, config.timezone);
      }

      // Setup completion handler if needed.
      if (config.completion_action && config.completion_action !== 'none') {
        setupCompletionHandler(container, displayElement, timer, config);
      }

      // Setup event triggers if enabled.
      if (config.enable_events) {
        setupEventTriggers(container, timer);
      }

      // Add timer control data attributes for external access.
      container.dataset.timerId = blockId;

    } catch (error) {
      console.error('Countdown: Failed to initialize timer', error);
    }
  }

  /**
   * Build CountdownTimer configuration from Drupal settings.
   *
   * @param {HTMLElement} displayElement
   *   The countdown display element.
   * @param {Object} config
   *   Configuration from Drupal settings.
   *
   * @returns {Object}
   *   CountdownTimer configuration object.
   */
  function buildTimerConfig(displayElement, config) {
    // Base configuration targeting the display element specifically.
    const timerConfig = {
      selector: '#' + displayElement.id,
      mode: config.timer_mode || 'countdown',
      precision: config.precision || 'seconds',
      autoStart: config.auto_start !== false,
      driftCompensation: config.drift_compensation !== false,
      debug: config.debug_mode || false
    };

    // Calculate duration based on mode.
    const targetTime = parseInt(config.timestamp, 10) * 1000;
    const now = Date.now();

    if (config.timer_mode === 'countdown') {
      // For countdown, calculate duration to target.
      timerConfig.duration = Math.max(0, Math.floor((targetTime - now) / 1000));
      // Apply user-defined offset for countdown mode.
      timerConfig.offset = parseInt(config.offset || 0, 10) * 1000;
    } else if (config.timer_mode === 'countup') {
      // For countup mode, offset handling varies by context.
      if (config.elapsed_from_start === true && config.offset) {
        // Elapsed mode: offset will be handled after timer creation.
        // The library doesn't support offset properly for unlimited countup.
        timerConfig.offset = 0;
      } else if (!config.is_elapsed_reset) {
        // Normal countup: calculate time since event.
        timerConfig.offset = Math.max(0, now - targetTime);
      } else {
        // Elapsed reset after countdown completion.
        timerConfig.offset = 0;
      }
    }

    // Apply display style configuration.
    const displayStyle = config.display_style || 'auto';

    if (displayStyle === 'verbose' || displayStyle === 'compact') {
      // Use built-in style renderers.
      timerConfig.style = displayStyle;
      timerConfig.styleOptions = {
        separator: config.separator || ', ',
        showZero: config.show_zero || false,
        maxUnits: parseInt(config.max_units || 4, 10)
      };

      // Add labels for verbose mode.
      if (displayStyle === 'verbose' && config.labels) {
        timerConfig.styleOptions.labels = config.labels;
      }
    } else if (displayStyle === 'custom' && config.custom_template) {
      // Use custom template through timer's formatTime.
      timerConfig.formatTime = function() {
        return config.custom_template;
      };
    }

    return timerConfig;
  }

  /**
   * Setup completion handler for countdown.
   *
   * @param {HTMLElement} container
   *   The countdown container element.
   * @param {HTMLElement} displayElement
   *   The countdown display element.
   * @param {CountdownTimer} timer
   *   The timer instance.
   * @param {Object} config
   *   Block configuration.
   */
  function setupCompletionHandler(container, displayElement, timer, config) {
    // Prevent multiple completion handler registrations.
    if (container.dataset.completionHandlerSet === 'true') {
      return;
    }

    timer.on('complete', function(time) {
      const action = config.completion_action;

      // Mark that completion has been handled.
      container.dataset.completionHandlerSet = 'true';

      switch (action) {
        case 'hide':
          // Fade out the entire countdown block container.
          container.style.transition = 'opacity 0.5s';
          container.style.opacity = '0';
          setTimeout(() => container.style.display = 'none', 500);
          break;

        case 'message':
          // Replace countdown display with completion message.
          if (config.completion_message) {
            displayElement.innerHTML = '<div class="countdown-complete">' +
              escapeHtml(config.completion_message) + '</div>';
          }
          break;

        case 'redirect':
          // Redirect to specified URL.
          if (config.completion_url) {
            window.location.href = config.completion_url;
          }
          break;

        case 'reload':
          // Reload the page.
          window.location.reload();
          break;

        case 'elapsed':
          // Switch to counting up from the event time.
          handleElapsedMode(container, displayElement, timer, config);
          break;

        case 'event':
          // Trigger custom event with data.
          triggerCustomEvent(container, config);
          break;
      }
    });
  }

  /**
   * Handle switching to elapsed time mode after countdown completes.
   *
   * @param {HTMLElement} container
   *   The countdown container element.
   * @param {HTMLElement} displayElement
   *   The countdown display element.
   * @param {CountdownTimer} timer
   *   The timer instance.
   * @param {Object} config
   *   Block configuration.
   */
  function handleElapsedMode(container, displayElement, timer, config) {
    // Destroy the current timer completely.
    timer.destroy();

    // Remove the old timer from the WeakMap.
    timerInstances.delete(container);

    // Build new configuration for unlimited countup mode.
    const elapsedConfig = {
      selector: '#' + displayElement.id,
      mode: 'countup',
      precision: config.precision || 'seconds',
      autoStart: true,
      driftCompensation: config.drift_compensation !== false,
      // Start from zero for elapsed time.
      offset: 0,
      // Critical: Do not set duration or target for unlimited counting.
      // Setting these would cause the timer to complete again.
      debug: config.debug_mode || false
    };

    // Apply the same display style configuration.
    const displayStyle = config.display_style || 'auto';

    if (displayStyle === 'verbose' || displayStyle === 'compact') {
      elapsedConfig.style = displayStyle;
      elapsedConfig.styleOptions = {
        separator: config.separator || ', ',
        showZero: config.show_zero || false,
        maxUnits: parseInt(config.max_units || 4, 10)
      };

      // Add labels for verbose mode.
      if (displayStyle === 'verbose' && config.labels) {
        elapsedConfig.styleOptions.labels = config.labels;
      }
    } else if (displayStyle === 'custom' && config.custom_template) {
      // Use custom template.
      elapsedConfig.formatTime = function() {
        return config.custom_template;
      };
    }

    // Create new timer instance for elapsed mode.
    const newTimer = new CountdownTimer(elapsedConfig);

    // Store the new timer instance.
    timerInstances.set(container, newTimer);

    // Update event info to show "since" instead of "until".
    const eventInfo = container.querySelector('.countdown-event-info');
    if (eventInfo && config.event_name) {
      // Use innerHTML to preserve links and HTML structure.
      const currentHTML = eventInfo.innerHTML;
      // Replace "until" with "since" while preserving HTML.
      const updatedHTML = currentHTML.replace(/\buntil\b/gi, 'since');
      eventInfo.innerHTML = updatedHTML;
    }

    // Mark container as being in elapsed mode.
    container.dataset.elapsedMode = 'true';

    // Trigger a custom event for the mode switch.
    container.dispatchEvent(new CustomEvent('countdown:elapsed', {
      detail: {
        blockId: container.dataset.blockId,
        timer: newTimer,
        originalTimestamp: container.dataset.originalTimestamp
      },
      bubbles: true
    }));

    // Re-enable event triggers if they were enabled.
    if (config.enable_events) {
      setupEventTriggers(container, newTimer);
    }

    // Log mode switch if debug is enabled.
    if (config.debug_mode) {
      console.log('Countdown: Switched to elapsed mode for block', container.dataset.blockId);
    }
  }

  /**
   * Setup event triggers for countdown updates.
   *
   * @param {HTMLElement} container
   *   The countdown container element.
   * @param {CountdownTimer} timer
   *   The timer instance.
   */
  function setupEventTriggers(container, timer) {
    // Check if events are already set up for this timer.
    const existingTimer = container.dataset.eventsSetupFor;
    const currentTimerId = timer._events ? timer._events._listeners.size : Math.random();

    if (existingTimer === String(currentTimerId)) {
      return;
    }

    // Mark this timer as having events set up.
    container.dataset.eventsSetupFor = String(currentTimerId);

    // Trigger tick events on the container for bubbling.
    timer.on('tick', function(time) {
      container.dispatchEvent(new CustomEvent('countdown:tick', {
        detail: { time: time, timer: timer },
        bubbles: true
      }));
    });

    // Trigger start event.
    timer.on('start', function(time) {
      container.dispatchEvent(new CustomEvent('countdown:start', {
        detail: { time: time, timer: timer },
        bubbles: true
      }));
    });

    // Trigger stop event.
    timer.on('stop', function(time) {
      container.dispatchEvent(new CustomEvent('countdown:stop', {
        detail: { time: time, timer: timer },
        bubbles: true
      }));
    });

    // Trigger pause event.
    timer.on('pause', function(time) {
      container.dispatchEvent(new CustomEvent('countdown:pause', {
        detail: { time: time, timer: timer },
        bubbles: true
      }));
    });

    // Trigger resume event.
    timer.on('resume', function(time) {
      container.dispatchEvent(new CustomEvent('countdown:resume', {
        detail: { time: time, timer: timer },
        bubbles: true
      }));
    });
  }

  /**
   * Trigger a custom completion event.
   *
   * @param {HTMLElement} container
   *   The countdown container element.
   * @param {Object} config
   *   Block configuration.
   */
  function triggerCustomEvent(container, config) {
    const eventName = config.completion_event_name || 'countdown:custom-complete';
    let eventData = { blockId: container.dataset.blockId };

    // Parse additional event data if provided.
    if (config.completion_event_data) {
      try {
        const parsedData = typeof config.completion_event_data === 'string'
          ? JSON.parse(config.completion_event_data)
          : config.completion_event_data;
        eventData = Object.assign(eventData, parsedData);
      } catch (e) {
        console.error('Countdown: Failed to parse event data', e);
      }
    }

    // Dispatch the custom event on the container.
    container.dispatchEvent(new CustomEvent(eventName, {
      detail: eventData,
      bubbles: true
    }));
  }

  /**
   * Add timezone display to countdown element.
   *
   * @param {HTMLElement} container
   *   The countdown container element.
   * @param {string} timezone
   *   The timezone identifier.
   */
  function addTimezoneDisplay(container, timezone) {
    // Check if timezone display already exists.
    if (container.querySelector('.countdown-timezone')) {
      return;
    }

    // Extract timezone abbreviation from identifier.
    const parts = timezone.split('/');
    const abbr = parts[parts.length - 1].replace(/_/g, ' ');

    // Find the best place to insert timezone.
    const eventInfo = container.querySelector('.countdown-event-info');
    const targetElement = eventInfo || container.querySelector('.countdown-realtime-content');

    if (targetElement) {
      // Create and append timezone indicator.
      const span = document.createElement('span');
      span.className = 'countdown-timezone';
      span.textContent = ' (' + abbr + ')';

      // Append to event info if it exists, otherwise to content.
      if (eventInfo) {
        eventInfo.appendChild(span);
      } else {
        targetElement.appendChild(span);
      }
    }
  }

  /**
   * Escape HTML to prevent XSS.
   *
   * @param {string} text
   *   Text to escape.
   *
   * @returns {string}
   *   Escaped HTML string.
   */
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  /**
   * Countdown utility namespace.
   *
   * @namespace
   */
  Drupal.countdown = Drupal.countdown || {};

  /**
   * Get timer instance by container or block ID.
   *
   * @param {string|HTMLElement} selector
   *   Block ID or container element.
   *
   * @returns {CountdownTimer|undefined}
   *   Timer instance or undefined if not found.
   */
  Drupal.countdown.getTimer = function(selector) {
    let container;

    if (typeof selector === 'string') {
      // Support both with and without 'countdown-' prefix.
      const id = selector.startsWith('countdown-') ? selector : 'countdown-' + selector;
      container = document.getElementById(id);
    } else {
      container = selector;
    }

    return container ? timerInstances.get(container) : undefined;
  };

  /**
   * Control a timer by block ID.
   *
   * @param {string} blockId
   *   The block ID.
   * @param {string} action
   *   The action to perform (start, stop, pause, resume, reset).
   * @param {Object} [options]
   *   Optional parameters for the action.
   *
   * @returns {boolean}
   *   True if action was performed, false otherwise.
   */
  Drupal.countdown.controlTimer = function(blockId, action, options) {
    const timer = this.getTimer(blockId);

    if (!timer) {
      console.warn('Countdown: Timer not found for block', blockId);
      return false;
    }

    // Execute the requested action.
    switch (action) {
      case 'start':
        timer.start(options);
        break;
      case 'stop':
        timer.stop();
        break;
      case 'pause':
        timer.pause();
        break;
      case 'resume':
        timer.resume();
        break;
      case 'reset':
        timer.reset(options);
        break;
      default:
        console.warn('Countdown: Unknown action', action);
        return false;
    }

    return true;
  };

  /**
   * Get all active timer instances.
   *
   * @returns {Array}
   *   Array of objects with blockId and timer instance.
   */
  Drupal.countdown.getAllTimers = function() {
    const timers = [];
    const containers = document.querySelectorAll('.countdown-timer-container[data-countdown-initialized]');

    containers.forEach(function(container) {
      const timer = timerInstances.get(container);
      if (timer) {
        timers.push({
          blockId: container.dataset.blockId,
          container: container,
          timer: timer
        });
      }
    });

    return timers;
  };

  /**
   * Check if a timer is in elapsed mode.
   *
   * @param {string} blockId
   *   The block ID to check.
   *
   * @returns {boolean}
   *   True if in elapsed mode, false otherwise.
   */
  Drupal.countdown.isElapsedMode = function(blockId) {
    const container = document.getElementById('countdown-' + blockId);
    return container ? container.dataset.elapsedMode === 'true' : false;
  };

})(Drupal, drupalSettings);
