<?php

namespace Drupal\countdown\Plugin\Block;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Datetime\TimeZoneFormHelper;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Countdown' Block.
 *
 * @Block(
 *   id = "countdown_block",
 *   admin_label = @Translation("Countdown"),
 * )
 */
class Countdown extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatter
   */
  protected $dateFormatter;

  /**
   * Constructs a new Countdown block instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Datetime\DateFormatter $date_formatter
   *   The date formatter service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DateFormatter $date_formatter) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    // Provide complete default configuration for backward compatibility.
    return [
      // Original fields from 8.x-1.x.
      'event_name' => '',
      'event_link' => '',
      'timestamp' => time(),
      'accuracy' => 's',

      // New fields with safe defaults (static mode preserves old behavior).
      'render_mode' => 'realtime',
      'timer_mode' => 'countdown',
      'precision' => 'seconds',
      'display_style' => 'auto',
      'custom_template' => '',
      'separator' => ', ',
      'show_zero' => FALSE,
      'max_units' => 4,
      'offset' => 0,
      'auto_start' => TRUE,
      'drift_compensation' => TRUE,
      'enable_events' => FALSE,
      'debug_mode' => FALSE,
      'completion_action' => 'none',
      'completion_message' => '',
      'completion_url' => '',
      'completion_event_name' => 'countdown:custom-complete',
      'completion_event_data' => '',
      'timezone' => date_default_timezone_get(),
      'show_timezone' => FALSE,

      // Legacy support for display_format field.
      'display_format' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * Get configuration value with fallback to default.
   *
   * This ensures backward compatibility for blocks created before update.
   *
   * @param string $key
   *   The configuration key.
   *
   * @return mixed
   *   The configuration value or default.
   */
  protected function getConfigValue($key) {
    $config = $this->getConfiguration();
    $defaults = $this->defaultConfiguration();

    // Handle legacy url to event_link migration.
    if ($key === 'event_link' && empty($config['event_link']) && !empty($config['url'])) {
      return $config['url'];
    }

    // Handle legacy display_format to display_style migration.
    if ($key === 'display_style' && empty($config['display_style']) && !empty($config['display_format'])) {
      // Map old display_format values to new display_style.
      $format_map = [
        'default' => 'auto',
        'verbose' => 'verbose',
        'compact' => 'compact',
      ];
      return $format_map[$config['display_format']] ?? 'auto';
    }

    // Return existing value or default.
    return $config[$key] ?? $defaults[$key] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $time = time();
    $block_id = $this->configuration['id'] ?? uniqid('countdown_');

    // Get configuration with backward compatibility.
    $event_name = $this->getConfigValue('event_name');
    $url = $this->getConfigValue('event_link') ?: '';
    $render_mode = $this->getConfigValue('render_mode');
    $countdown_timestamp = $this->getConfigValue('timestamp') ?: $time;

    // Ensure render_mode has a valid value for old blocks.
    if (!in_array($render_mode, ['static', 'realtime'])) {
      $render_mode = 'static';
    }

    // Create event link if URL is provided and event name exists.
    $formatted_event_name = '';
    if (!empty($event_name)) {
      $formatted_event_name = Html::escape($event_name);
      if (!empty($url)) {
        try {
          if (UrlHelper::isExternal($url)) {
            $formatted_event_name = Link::fromTextAndUrl($event_name, Url::fromUri($url))->toString();
          }
          else {
            if ($url == "<front>") {
              $url = "/";
            }
            $formatted_event_name = Link::fromTextAndUrl($event_name, Url::fromUri('internal:/' . ltrim($url, '/')))->toString();
          }
        }
        catch (\Exception $e) {
          // If URL is invalid, just use the plain event name.
          $formatted_event_name = Html::escape($event_name);
        }
      }
    }

    // Build render array based on mode.
    if ($render_mode === 'static') {
      // Use legacy static rendering for backward compatibility.
      return $this->buildStaticCountdown($formatted_event_name, $event_name, $countdown_timestamp, $time);
    }
    else {
      // Use new real-time rendering.
      return $this->buildRealtimeCountdown($formatted_event_name, $event_name, $countdown_timestamp, $block_id);
    }
  }

  /**
   * Build static countdown (legacy PHP-based rendering).
   *
   * This method preserves the exact original behavior for existing blocks.
   *
   * @param string $formatted_event_name
   *   The formatted event name with optional link.
   * @param string $raw_event_name
   *   The raw event name without formatting.
   * @param int $countdown_timestamp
   *   The target timestamp.
   * @param int $time
   *   The current timestamp.
   *
   * @return array
   *   The render array.
   */
  protected function buildStaticCountdown($formatted_event_name, $raw_event_name, $countdown_timestamp, $time) {
    $difference = $countdown_timestamp - $time;
    $completion_action = $this->getConfigValue('completion_action');
    $timer_mode = $this->getConfigValue('timer_mode');

    // Check if event has completed and handle completion actions.
    $is_completed = FALSE;
    if ($timer_mode === 'countdown' && $difference <= 0) {
      $is_completed = TRUE;
    }
    elseif ($timer_mode === 'countup' && $difference >= 0) {
      // For countup, we never complete in static mode.
      $is_completed = FALSE;
    }

    // Handle completion actions if completed.
    if ($is_completed && $completion_action && $completion_action !== 'none') {
      $completion_result = $this->handleStaticCompletion($completion_action, $formatted_event_name, $raw_event_name);
      if ($completion_result !== FALSE) {
        return $completion_result;
      }
    }

    // Get accuracy with backward compatibility.
    $accuracy = $this->getConfigValue('accuracy');
    if (!in_array($accuracy, ['d', 'h', 'm', 's'])) {
      $accuracy = 's';
    }

    // For 'none' action or when not completed, show countdown.
    // For 'none' action when completed, show zeros instead of negative.
    if ($completion_action === 'none' && $is_completed) {
      // Show zeros when completed with 'none' action.
      $days_left = $this->formatPlural(0, '1 day', '@count days');
      $hrs_left = ($accuracy == 'h' || $accuracy == 'm' || $accuracy == 's')
        ? $this->formatPlural(0, ', 1 hour', ', @count hours')
        : '';
      $min_left = ($accuracy == 'm' || $accuracy == 's')
        ? $this->formatPlural(0, ', 1 minute', ', @count minutes')
        : '';
      $secs_left = ($accuracy == 's')
        ? $this->formatPlural(0, ', 1 second', ', @count seconds')
        : '';
    }
    elseif ($completion_action === 'elapsed' && $is_completed) {
      // Switch to counting up from the event time.
      $elapsed_difference = abs($difference);
      $days_left = floor($elapsed_difference / 60 / 60 / 24);
      $hrs_left = floor(($elapsed_difference - $days_left * 60 * 60 * 24) / 60 / 60);
      $min_left = floor(($elapsed_difference - $days_left * 60 * 60 * 24 - $hrs_left * 60 * 60) / 60);
      $secs_left = floor(($elapsed_difference - $days_left * 60 * 60 * 24 - $hrs_left * 60 * 60 - $min_left * 60));

      // Format with plurals for elapsed time.
      $days_left = $this->formatPlural($days_left, '1 day', '@count days');
      $hrs_left = ($accuracy == 'h' || $accuracy == 'm' || $accuracy == 's')
        ? $this->formatPlural($hrs_left, ', 1 hour', ', @count hours')
        : '';
      $min_left = ($accuracy == 'm' || $accuracy == 's')
        ? $this->formatPlural($min_left, ', 1 minute', ', @count minutes')
        : '';
      $secs_left = ($accuracy == 's')
        ? $this->formatPlural($secs_left, ', 1 second', ', @count seconds')
        : '';

      // Change direction text for elapsed time.
      if (!empty($formatted_event_name)) {
        $formatted_event_name = $this->t('since @event_name.', ['@event_name' => $formatted_event_name]);
      }
    }
    else {
      // Normal countdown/countup calculation.
      $abs_difference = abs($difference);
      $days_left = floor($abs_difference / 60 / 60 / 24);
      $hrs_left = floor(($abs_difference - $days_left * 60 * 60 * 24) / 60 / 60);
      $min_left = floor(($abs_difference - $days_left * 60 * 60 * 24 - $hrs_left * 60 * 60) / 60);
      $secs_left = floor(($abs_difference - $days_left * 60 * 60 * 24 - $hrs_left * 60 * 60 - $min_left * 60));

      // Format with plurals (preserve original formatting).
      $days_left = $this->formatPlural($days_left, '1 day', '@count days');
      $hrs_left = ($accuracy == 'h' || $accuracy == 'm' || $accuracy == 's')
        ? $this->formatPlural($hrs_left, ', 1 hour', ', @count hours')
        : '';
      $min_left = ($accuracy == 'm' || $accuracy == 's')
        ? $this->formatPlural($min_left, ', 1 minute', ', @count minutes')
        : '';
      $secs_left = ($accuracy == 's')
        ? $this->formatPlural($secs_left, ', 1 second', ', @count seconds')
        : '';

      // Format event name with direction (preserve original behavior).
      // Only add direction if event name exists.
      if (!empty($formatted_event_name)) {
        if ($timer_mode === 'countup' || $difference < 0) {
          $formatted_event_name = $this->t('since @event_name.', ['@event_name' => $formatted_event_name]);
        }
        else {
          $formatted_event_name = $this->t('until @event_name.', ['@event_name' => $formatted_event_name]);
        }
      }
    }

    // Return using original theme structure for backward compatibility.
    return [
      '#theme' => 'countdown',
      '#cache' => ['max-age' => 0],
      '#accuracy' => $accuracy,
      '#event_link' => $this->getConfigValue('event_link'),
      '#event_name' => $formatted_event_name,
      '#days_left' => $days_left,
      '#hrs_left' => $hrs_left,
      '#min_left' => $min_left,
      '#secs_left' => $secs_left,
      '#render_mode' => 'static',
      '#attached' => [
        'library' => [
          'countdown/block',
        ],
        'drupalSettings' => [
          'countdown' => [
            'block' => [
              'accuracy' => $accuracy,
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * Handles completion actions for static mode.
   *
   * @param string $action
   *   The completion action.
   * @param string $formatted_event_name
   *   The formatted event name with HTML.
   * @param string $raw_event_name
   *   The raw event name without HTML.
   *
   * @return array|false
   *   The render array for completion or FALSE to continue normal display.
   */
  protected function handleStaticCompletion($action, $formatted_event_name, $raw_event_name) {
    switch ($action) {
      case 'none':
        // Return FALSE to show the countdown at zero.
        return FALSE;

      case 'hide':
        // Return empty render array to hide the block.
        return [
          '#markup' => '',
          '#cache' => ['max-age' => 0],
        ];

      case 'message':
        // Show the configured completion message.
        $message = $this->getConfigValue('completion_message');
        if (empty($message)) {
          $message = $this->t('The event has started!');
        }
        return [
          '#markup' => '<div class="countdown-completion-message">' . Html::escape($message) . '</div>',
          '#cache' => ['max-age' => 0],
        ];

      case 'redirect':
        // In static mode, we cannot do server-side redirect after render.
        // Instead, we use a meta refresh tag for automatic redirection.
        $url = $this->getConfigValue('completion_url');
        if (!empty($url)) {
          try {
            // Validate and build the URL.
            if (UrlHelper::isExternal($url)) {
              $redirect_url = $url;
            }
            else {
              if ($url == "<front>") {
                $url = "/";
              }
              $redirect_url = Url::fromUri('internal:/' . ltrim($url, '/'))->toString();
            }

            // Use meta refresh for automatic redirect after 1 second.
            return [
              '#type' => 'html_tag',
              '#tag' => 'div',
              '#value' => $this->t('Redirecting to event page...'),
              '#attributes' => [
                'class' => ['countdown-completion-redirect'],
              ],
              '#attached' => [
                'html_head' => [
                  [
                    [
                      '#tag' => 'meta',
                      '#attributes' => [
                        'http-equiv' => 'refresh',
                        'content' => '1;url=' . $redirect_url,
                      ],
                    ],
                    'countdown_redirect_' . $this->configuration['id'],
                  ],
                ],
              ],
              '#cache' => ['max-age' => 0],
            ];
          }
          catch (\Exception $e) {
            // If URL is invalid, show error message.
            return [
              '#markup' => '<div class="countdown-completion-error">' . Html::escape($this->t('Invalid redirect URL configured.')) . '</div>',
              '#cache' => ['max-age' => 0],
            ];
          }
        }
        // If no URL configured, fall back to message.
        return $this->handleStaticCompletion('message', $formatted_event_name, $raw_event_name);

      case 'reload':
        // Use meta refresh to reload the page after 1 second.
        return [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#value' => $this->t('Reloading page...'),
          '#attributes' => [
            'class' => ['countdown-completion-reload'],
          ],
          '#attached' => [
            'html_head' => [
              [
                [
                  '#tag' => 'meta',
                  '#attributes' => [
                    'http-equiv' => 'refresh',
                    'content' => '1',
                  ],
                ],
                'countdown_reload_' . $this->configuration['id'],
              ],
            ],
          ],
          '#cache' => ['max-age' => 0],
        ];

      case 'elapsed':
        // Return FALSE to show elapsed time (handled in main method).
        return FALSE;

      case 'event':
        // In static mode, we cannot trigger JavaScript events.
        // Show a message indicating the event would have been triggered.
        $event_name = $this->getConfigValue('completion_event_name');
        $event_data = $this->getConfigValue('completion_event_data');

        // For static mode, we can embed JavaScript to trigger the event.
        $js_code = '';
        if (!empty($event_name)) {
          $event_detail = [];
          if (!empty($event_data)) {
            $decoded = json_decode($event_data, TRUE);
            if (json_last_error() === JSON_ERROR_NONE) {
              $event_detail = $decoded;
            }
          }

          // Create JavaScript to trigger the event on page load.
          $js_code = sprintf(
            "(function() {
              var event = new CustomEvent('%s', { detail: %s, bubbles: true });
              document.dispatchEvent(event);
            })();",
            Html::escape($event_name),
            json_encode($event_detail)
          );
        }

        return [
          '#markup' => '<div class="countdown-completion-event" data-event="' . Html::escape($event_name) . '">' . Html::escape($this->t('Event completed: @event', ['@event' => $raw_event_name ?: $this->t('Countdown')])) . '</div>',
          '#attached' => !empty($js_code) ? [
            'html_head' => [
              [
                [
                  '#tag' => 'script',
                  '#value' => $js_code,
                  '#attributes' => ['type' => 'text/javascript'],
                ],
                'countdown_event_trigger_' . $this->configuration['id'],
              ],
            ],
          ] : [],
          '#cache' => ['max-age' => 0],
        ];

      default:
        // Unknown action, fall back to 'none'.
        return FALSE;
    }
  }

  /**
   * Build real-time countdown (new JavaScript-based rendering).
   *
   * @param string $formatted_event_name
   *   The formatted event name with link if URL provided.
   * @param string $raw_event_name
   *   The raw event name without formatting.
   * @param int $countdown_timestamp
   *   The target timestamp.
   * @param string $block_id
   *   The block ID for JavaScript initialization.
   *
   * @return array
   *   The render array.
   */
  protected function buildRealtimeCountdown($formatted_event_name, $raw_event_name, $countdown_timestamp, $block_id) {
    $time = time();
    $difference = $countdown_timestamp - $time;

    // Determine timer mode with intelligent default.
    $timer_mode = $this->getConfigValue('timer_mode');
    if (empty($timer_mode)) {
      // Auto-detect based on time difference.
      $timer_mode = $difference < 0 ? 'countup' : 'countdown';
    }

    // Get completion action for mode adjustment.
    $completion_action = $this->getConfigValue('completion_action');

    // For elapsed mode, ensure we start in countdown if future event.
    if ($completion_action === 'elapsed' && $difference > 0) {
      // Force countdown mode for future events with elapsed completion.
      $timer_mode = 'countdown';
    }

    // Format event name with direction for template display.
    $event_with_direction = '';
    if (!empty($formatted_event_name)) {
      // Check if we should show elapsed time immediately.
      if ($completion_action === 'elapsed' && $difference <= 0) {
        // Event has passed and elapsed mode is set.
        $event_with_direction = $this->t('since @event', ['@event' => $formatted_event_name]);
      }
      elseif ($timer_mode === 'countdown') {
        $event_with_direction = $this->t('until @event', ['@event' => $formatted_event_name]);
      }
      else {
        $event_with_direction = $this->t('since @event', ['@event' => $formatted_event_name]);
      }
    }

    // Build JavaScript configuration.
    $js_config = [
      'render_mode' => 'realtime',
      'timer_mode' => $timer_mode,
      'timestamp' => $countdown_timestamp,
      'event_name' => $raw_event_name,
      'precision' => $this->getConfigValue('precision'),
      'display_style' => $this->getConfigValue('display_style'),
      'custom_template' => $this->getConfigValue('custom_template'),
      'auto_start' => $this->getConfigValue('auto_start'),
      'drift_compensation' => $this->getConfigValue('drift_compensation'),
      'offset' => ($completion_action === 'elapsed' && $difference <= 0)
        ? 0
        : (int) $this->getConfigValue('offset'),
      'debug_mode' => $this->getConfigValue('debug_mode'),
      'timezone' => $this->getConfigValue('timezone'),
      'show_timezone' => $this->getConfigValue('show_timezone'),
    ];

    // Determine display format based on style and template.
    $display_style = $this->getConfigValue('display_style');

    if ($display_style === 'custom' && !empty($js_config['custom_template'])) {
      // Use custom template.
      $js_config['display_format'] = 'custom';
      $js_config['format_template'] = $js_config['custom_template'];
    }
    elseif (in_array($display_style, ['verbose', 'compact'])) {
      // Use predefined style.
      $js_config['display_format'] = $display_style;
      $js_config['separator'] = $this->getConfigValue('separator');
      $js_config['show_zero'] = $this->getConfigValue('show_zero');
      $js_config['max_units'] = (int) $this->getConfigValue('max_units');
    }
    else {
      // Auto mode: use default format based on precision.
      $js_config['display_format'] = 'auto';
    }

    // Add completion configuration if specified.
    $completion_action = $this->getConfigValue('completion_action');
    if (!empty($completion_action) && $completion_action !== 'none') {
      $js_config['completion_action'] = $completion_action;

      // Handle elapsed mode for past events.
      if ($completion_action === 'elapsed' && $difference <= 0) {
        // Event already passed, start in countup mode immediately.
        $js_config['timer_mode'] = 'countup';
        // Calculate elapsed time since event.
        $js_config['offset'] = abs($difference) * 1000;
        // Mark this as elapsed mode from start for JS.
        $js_config['elapsed_from_start'] = TRUE;
      }

      if ($completion_action === 'message') {
        $js_config['completion_message'] = $this->getConfigValue('completion_message') ?: $this->t('The event has ended.');
      }
      elseif ($completion_action === 'redirect') {
        $js_config['completion_url'] = $this->getConfigValue('completion_url');
      }
      elseif ($completion_action === 'event') {
        $js_config['completion_event_name'] = $this->getConfigValue('completion_event_name') ?: 'countdown:custom-complete';
        $event_data = $this->getConfigValue('completion_event_data');

        // Validate and parse JSON data if provided.
        if (!empty($event_data)) {
          $decoded = json_decode($event_data, TRUE);
          if (json_last_error() === JSON_ERROR_NONE) {
            $js_config['completion_event_data'] = $decoded;
          }
        }
      }
    }

    // Enable events if configured.
    $js_config['enable_events'] = $this->getConfigValue('enable_events');

    // Generate fallback content for noscript.
    $fallback = $this->generateFallbackContent($formatted_event_name, $countdown_timestamp);

    // Use the theme with proper variables for realtime mode.
    return [
      '#theme' => 'countdown',
      '#cache' => ['max-age' => 0],
      '#block_id' => $block_id,
      '#render_mode' => 'realtime',
      '#event_name' => $event_with_direction,
      '#event_link' => $this->getConfigValue('event_link'),
      '#attributes' => [
        'id' => 'countdown-' . $block_id,
        'class' => ['countdown-timer-container'],
        'data-block-id' => $block_id,
      ],
      '#attached' => [
        'library' => [
          'countdown/timer',
          'countdown/integration',
        ],
        'drupalSettings' => [
          'countdown' => [
            'countdown_' . $block_id => $js_config,
          ],
        ],
      ],
      // Set placeholder values for template compatibility.
      '#accuracy' => '',
      '#days_left' => '',
      '#hrs_left' => '',
      '#min_left' => '',
      '#secs_left' => '',

      // Add the fallback content as a variable for template.
      '#fallback_content' => $fallback,
    ];
  }

  /**
   * Generate fallback content for users without JavaScript.
   *
   * @param string $event_name
   *   The formatted event name.
   * @param int $countdown_timestamp
   *   The target timestamp.
   *
   * @return string
   *   HTML fallback content.
   */
  protected function generateFallbackContent($event_name, $countdown_timestamp) {
    $time = time();
    $difference = $countdown_timestamp - $time;

    // Build time text.
    $time_text = $this->dateFormatter->formatInterval(abs($difference));

    // Build full text based on whether event name exists.
    if (!empty($event_name)) {
      if ($difference < 0) {
        $text = $this->t('@time since @event', [
          '@time' => $time_text,
          '@event' => $event_name,
        ]);
      }
      else {
        $text = $this->t('@time until @event', [
          '@time' => $time_text,
          '@event' => $event_name,
        ]);
      }
    }
    else {
      // Just show the time without event context.
      $text = $time_text;
    }

    return '<noscript><div class="countdown-static">' . Html::escape($text) . '</div></noscript>';
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $time = time();

    // Add library for admin form styling.
    $form['#attached']['library'][] = 'countdown/admin';

    // Event settings.
    $form['event_settings'] = [
      '#title' => $this->t('Event Settings'),
      '#type' => 'details',
      '#open' => TRUE,
      '#description' => $this->t("The event you're counting to or from."),
    ];

    $form['event_settings']['event'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['event-settings']],
    ];

    $form['event_settings']['event']['event_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event Name'),
      '#title_display' => 'invisible',
      '#default_value' => $this->getConfigValue('event_name'),
      '#size' => 64,
      '#maxlength' => 256,
      '#required' => FALSE,
      '#placeholder' => $this->t('Event name ...'),
      '#attributes' => ['class' => ['event__name']],
    ];

    $form['event_settings']['event']['event_link'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event URL'),
      '#title_display' => 'invisible',
      '#default_value' => $this->getConfigValue('event_link'),
      '#size' => 64,
      '#maxlength' => 256,
      '#placeholder' => $this->t('Enter a optional link to more information ...'),
      '#attributes' => ['class' => ['event__link']],
      '#description' => $this->t('Use %front for the front page, or enter an internal path like /node/123 or external URL. Leave blank to show only the event name.', [
        '%front' => '<front>',
      ]),
    ];

    // Display settings.
    $form['display_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Settings'),
    ];

    $form['display_settings']['render_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Render Mode'),
      '#options' => [
        'static' => $this->t('Server-side (Updates on page refresh)'),
        'realtime' => $this->t('Real-time JavaScript countdown'),
      ],
      '#default_value' => $this->getConfigValue('render_mode'),
      '#description' => $this->t('Choose between lightweight server-side rendering or dynamic JavaScript countdown.'),
    ];

    // Static mode settings (preserve original field).
    $form['display_settings']['accuracy'] = [
      '#type' => 'radios',
      '#title' => $this->t('Display Accuracy'),
      '#options' => [
        'd' => $this->t('Days'),
        'h' => $this->t('Hours'),
        'm' => $this->t('Minutes'),
        's' => $this->t('Seconds'),
      ],
      '#default_value' => $this->getConfigValue('accuracy'),
      '#description' => $this->t('The smallest unit to display. For example, selecting "hours" will show days and hours.'),
      '#states' => [
        'visible' => [
          ':input[name="settings[display_settings][render_mode]"]' => ['value' => 'static'],
        ],
      ],
    ];

    // JavaScript mode settings.
    $form['js_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('JavaScript Timer Settings'),
      '#states' => [
        'visible' => [
          ':input[name="settings[display_settings][render_mode]"]' => ['value' => 'realtime'],
        ],
      ],
    ];

    $form['js_settings']['timer_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Timer Mode'),
      '#options' => [
        'countdown' => $this->t('Count down to event'),
        'countup' => $this->t('Count up from event'),
      ],
      '#default_value' => $this->getConfigValue('timer_mode'),
      '#description' => $this->t('Choose whether to count down to a future event or count up from a past event.'),
    ];

    $form['js_settings']['precision'] = [
      '#type' => 'select',
      '#title' => $this->t('Time Precision'),
      '#options' => [
        'minutes' => $this->t('Minutes (HH:MM)'),
        'seconds' => $this->t('Seconds (HH:MM:SS)'),
        'tenths' => $this->t('Tenths of a second (HH:MM:SS.m)'),
        'hundredths' => $this->t('Hundredths (HH:MM:SS.mm)'),
        'milliseconds' => $this->t('Milliseconds (HH:MM:SS.mmm)'),
      ],
      '#default_value' => $this->getConfigValue('precision'),
      '#description' => $this->t('Select the precision level for time display and update frequency.'),
    ];

    $form['js_settings']['display_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Display Style'),
      '#options' => [
        'auto' => $this->t('Automatic (based on precision)'),
        'verbose' => $this->t('Verbose (3 days, 2 hours, 1 minute)'),
        'compact' => $this->t('Compact (3d 2h 1m)'),
        'custom' => $this->t('Custom template'),
      ],
      '#default_value' => $this->getConfigValue('display_style'),
      '#description' => $this->t('Choose how to display the countdown. Auto uses the default format for the selected precision.'),
    ];

    $form['js_settings']['custom_template'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Format Template'),
      '#default_value' => $this->getConfigValue('custom_template'),
      '#size' => 60,
      '#maxlength' => 128,
      '#description' => $this->t('Custom time format. Available tokens: DD (days), D (days no pad), HH (hours), H (hours no pad), MM (minutes), M (minutes no pad), SS (seconds), S (seconds no pad), mmm (milliseconds), mm (hundredths), m (tenths). Example: "DD days, HH:MM:SS remaining"'),
      '#states' => [
        'visible' => [
          ':input[name="settings[js_settings][display_style]"]' => ['value' => 'custom'],
        ],
        'required' => [
          ':input[name="settings[js_settings][display_style]"]' => ['value' => 'custom'],
        ],
      ],
    ];

    // Format customization.
    $form['js_settings']['format_options'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Format Options'),
      '#states' => [
        'visible' => [
          [':input[name="settings[js_settings][display_style]"]' => ['value' => 'verbose']],
          [':input[name="settings[js_settings][display_style]"]' => ['value' => 'compact']],
        ],
      ],
    ];

    $form['js_settings']['format_options']['separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Unit Separator'),
      '#default_value' => $this->getConfigValue('separator'),
      '#size' => 10,
      '#description' => $this->t('Text between time units.'),
    ];

    $form['js_settings']['format_options']['show_zero'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show zero values'),
      '#default_value' => $this->getConfigValue('show_zero'),
      '#description' => $this->t('Display units even when their value is zero.'),
    ];

    $form['js_settings']['format_options']['max_units'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum units to display'),
      '#default_value' => $this->getConfigValue('max_units'),
      '#min' => 1,
      '#max' => 7,
      '#description' => $this->t('Limit the number of time units shown.'),
    ];

    // Enhanced Target date/time with improved layout.
    $form['target_time'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Target Date/Time'),
      '#description' => $this->t('Select a date relative to server time: %s', [
        '%s' => $this->dateFormatter->format($time),
      ]),
      '#attributes' => [
        'class' => ['countdown-datetime-fieldset'],
      ],
    ];

    $timestamp = $this->getConfigValue('timestamp') ?: $time;

    // Date row container.
    $form['target_time']['date_row'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['countdown-datetime-row', 'countdown-date-row'],
      ],
    ];

    $form['target_time']['date_row']['month'] = [
      '#type' => 'select',
      '#title' => $this->t('Month'),
      '#default_value' => (int) date('n', $timestamp),
      '#options' => [
        1 => $this->t('January'),
        2 => $this->t('February'),
        3 => $this->t('March'),
        4 => $this->t('April'),
        5 => $this->t('May'),
        6 => $this->t('June'),
        7 => $this->t('July'),
        8 => $this->t('August'),
        9 => $this->t('September'),
        10 => $this->t('October'),
        11 => $this->t('November'),
        12 => $this->t('December'),
      ],
      '#attributes' => [
        'class' => ['countdown-month-select'],
      ],
      '#prefix' => '<div class="countdown-datetime-item countdown-month-item">',
      '#suffix' => '</div>',
    ];

    $form['target_time']['date_row']['day'] = [
      '#type' => 'number',
      '#title' => $this->t('Day'),
      '#default_value' => (int) date('j', $timestamp),
      '#min' => 1,
      '#max' => 31,
      '#attributes' => [
        'class' => ['countdown-day-input'],
      ],
      '#prefix' => '<div class="countdown-datetime-item countdown-day-item">',
      '#suffix' => '</div>',
    ];

    $form['target_time']['date_row']['year'] = [
      '#type' => 'number',
      '#title' => $this->t('Year'),
      '#default_value' => (int) date('Y', $timestamp),
      '#min' => 1970,
      '#max' => 2100,
      '#attributes' => [
        'class' => ['countdown-year-input'],
      ],
      '#prefix' => '<div class="countdown-datetime-item countdown-year-item">',
      '#suffix' => '</div>',
    ];

    // Time row container.
    $form['target_time']['time_row'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['countdown-datetime-row', 'countdown-time-row'],
      ],
    ];

    $form['target_time']['time_row']['hour'] = [
      '#type' => 'number',
      '#title' => $this->t('Hour'),
      '#default_value' => (int) date('G', $timestamp),
      '#min' => 0,
      '#max' => 23,
      '#attributes' => [
        'class' => ['countdown-hour-input'],
      ],
      '#prefix' => '<div class="countdown-datetime-item countdown-hour-item">',
      '#suffix' => '</div>',
    ];

    $form['target_time']['time_row']['min'] = [
      '#type' => 'number',
      '#title' => $this->t('Minute'),
      '#default_value' => (int) date('i', $timestamp),
      '#min' => 0,
      '#max' => 59,
      '#attributes' => [
        'class' => ['countdown-minute-input'],
      ],
      '#prefix' => '<div class="countdown-datetime-item countdown-minute-item">',
      '#suffix' => '</div>',
    ];

    $form['target_time']['time_row']['sec'] = [
      '#type' => 'number',
      '#title' => $this->t('Seconds'),
      '#default_value' => (int) date('s', $timestamp),
      '#min' => 0,
      '#max' => 59,
      '#attributes' => [
        'class' => ['countdown-second-input'],
      ],
      '#prefix' => '<div class="countdown-datetime-item countdown-second-item">',
      '#suffix' => '</div>',
    ];

    // Timezone settings.
    $form['target_time']['timezone_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Timezone Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="settings[display_settings][render_mode]"]' => ['value' => 'realtime'],
        ],
      ],
    ];

    $form['target_time']['timezone_settings']['timezone'] = [
      '#type' => 'select',
      '#title' => $this->t('Timezone'),
      '#default_value' => $this->getConfigValue('timezone'),
      '#options' => TimeZoneFormHelper::getOptionsList(),
      '#description' => $this->t('The timezone for the target date. This affects how the countdown is calculated.'),
    ];

    $form['target_time']['timezone_settings']['show_timezone'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display timezone'),
      '#default_value' => $this->getConfigValue('show_timezone'),
      '#description' => $this->t('Show the timezone abbreviation in the countdown display.'),
    ];

    // Advanced settings.
    $form['advanced_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="settings[display_settings][render_mode]"]' => ['value' => 'realtime'],
        ],
      ],
    ];

    $form['advanced_settings']['offset'] = [
      '#type' => 'number',
      '#title' => $this->t('Start Offset (seconds)'),
      '#default_value' => $this->getConfigValue('offset'),
      '#description' => $this->t('Offset the countdown start time by this many seconds.'),
    ];

    $form['advanced_settings']['auto_start'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-start timer'),
      '#default_value' => $this->getConfigValue('auto_start'),
      '#description' => $this->t('Automatically start the countdown when the page loads.'),
    ];

    $form['advanced_settings']['drift_compensation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable drift compensation'),
      '#default_value' => $this->getConfigValue('drift_compensation'),
      '#description' => $this->t('Compensate for timer drift to maintain accuracy over long periods.'),
    ];

    $form['advanced_settings']['enable_events'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable JavaScript events'),
      '#default_value' => $this->getConfigValue('enable_events'),
      '#description' => $this->t('Trigger custom JavaScript events for integration with other modules.'),
    ];

    $form['advanced_settings']['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug mode'),
      '#default_value' => $this->getConfigValue('debug_mode'),
      '#description' => $this->t('Log timer information to the browser console.'),
    ];

    // Completion actions.
    $form['completion'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Completion Actions'),
      '#states' => [
        'invisible' => [
          [
            ':input[name="settings[display_settings][render_mode]"]' => ['value' => 'realtime'],
            ':input[name="settings[js_settings][timer_mode]"]' => ['value' => 'countup'],
          ],
          [
            ':input[name="settings[display_settings][render_mode]"]' => ['value' => 'static'],
            ':input[name="settings[js_settings][timer_mode]"]' => ['value' => 'countup'],
          ],
        ],
      ],
    ];

    $form['completion']['completion_action'] = [
      '#type' => 'select',
      '#title' => $this->t('When countdown completes'),
      '#options' => [
        'none' => $this->t('Do nothing (stop at zero)'),
        'hide' => $this->t('Hide the countdown'),
        'message' => $this->t('Show a message'),
        'redirect' => $this->t('Redirect to URL'),
        'reload' => $this->t('Reload the page'),
        'elapsed' => $this->t('Switch to elapsed time (count-up)'),
        'event' => $this->t('Trigger custom event'),
      ],
      '#default_value' => $this->getConfigValue('completion_action'),
      '#description' => $this->t('Action to perform when the countdown reaches zero. Note: Some actions have limited functionality in static mode.'),
    ];

    $form['completion']['completion_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Completion message'),
      '#default_value' => $this->getConfigValue('completion_message') ?: $this->t('The event has started!'),
      '#size' => 60,
      '#states' => [
        'visible' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'message'],
        ],
        'required' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'message'],
        ],
      ],
    ];

    $form['completion']['completion_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Redirect URL'),
      '#default_value' => $this->getConfigValue('completion_url'),
      '#size' => 60,
      '#description' => $this->t('URL to redirect to when countdown completes. In static mode, uses meta refresh with 1 second delay.'),
      '#states' => [
        'visible' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'redirect'],
        ],
        'required' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'redirect'],
        ],
      ],
    ];

    $form['completion']['completion_event_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Event Name'),
      '#default_value' => $this->getConfigValue('completion_event_name') ?: 'countdown:custom-complete',
      '#size' => 40,
      '#description' => $this->t('JavaScript event name to trigger. Must be a valid event name.'),
      '#states' => [
        'visible' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'event'],
        ],
        'required' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'event'],
        ],
      ],
    ];

    $form['completion']['completion_event_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Event Data (JSON)'),
      '#default_value' => $this->getConfigValue('completion_event_data'),
      '#rows' => 3,
      '#description' => $this->t('Optional JSON data to pass with the event. Must be valid JSON format. Example: {"action": "show-modal", "id": 123}'),
      '#states' => [
        'visible' => [
          ':input[name="settings[completion][completion_action]"]' => ['value' => 'event'],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {

    // Basic settings.
    $this->setConfigurationValue('event_name', $form_state->getValue([
      'event_settings',
      'event',
      'event_name',
    ]));
    $this->setConfigurationValue('event_link', $form_state->getValue([
      'event_settings',
      'event',
      'event_link',
    ]));

    // Display settings.
    $this->setConfigurationValue('render_mode', $form_state->getValue([
      'display_settings',
      'render_mode',
    ]));
    $this->setConfigurationValue('accuracy', $form_state->getValue([
      'display_settings',
      'accuracy',
    ]));

    // Timer mode is saved regardless of render mode for consistency.
    $this->setConfigurationValue('timer_mode', $form_state->getValue([
      'js_settings',
      'timer_mode',
    ]));

    // JavaScript settings (only save if in realtime mode).
    $render_mode = $form_state->getValue([
      'display_settings',
      'render_mode',
    ]);
    if ($render_mode === 'realtime') {
      $this->setConfigurationValue('precision', $form_state->getValue([
        'js_settings',
        'precision',
      ]));
      $this->setConfigurationValue('display_style', $form_state->getValue([
        'js_settings',
        'display_style',
      ]));
      $this->setConfigurationValue('custom_template', $form_state->getValue([
        'js_settings',
        'custom_template',
      ]));

      // Format options.
      $this->setConfigurationValue('separator', $form_state->getValue([
        'js_settings',
        'format_options',
        'separator',
      ]));
      $this->setConfigurationValue('show_zero', $form_state->getValue([
        'js_settings',
        'format_options',
        'show_zero',
      ]));
      $this->setConfigurationValue('max_units', $form_state->getValue([
        'js_settings',
        'format_options',
        'max_units',
      ]));

      // Advanced settings.
      $this->setConfigurationValue('offset', $form_state->getValue([
        'advanced_settings',
        'offset',
      ]));
      $this->setConfigurationValue('auto_start', $form_state->getValue([
        'advanced_settings',
        'auto_start',
      ]));
      $this->setConfigurationValue('drift_compensation', $form_state->getValue([
        'advanced_settings',
        'drift_compensation',
      ]));
      $this->setConfigurationValue('enable_events', $form_state->getValue([
        'advanced_settings',
        'enable_events',
      ]));
      $this->setConfigurationValue('debug_mode', $form_state->getValue([
        'advanced_settings',
        'debug_mode',
      ]));

      // Timezone settings.
      $this->setConfigurationValue('timezone', $form_state->getValue([
        'target_time',
        'timezone_settings',
        'timezone',
      ]));
      $this->setConfigurationValue('show_timezone', $form_state->getValue([
        'target_time',
        'timezone_settings',
        'show_timezone',
      ]));
    }

    // Completion settings (save for both modes).
    $this->setConfigurationValue('completion_action', $form_state->getValue([
      'completion',
      'completion_action',
    ]));
    $this->setConfigurationValue('completion_message', $form_state->getValue([
      'completion',
      'completion_message',
    ]));
    $this->setConfigurationValue('completion_url', $form_state->getValue([
      'completion',
      'completion_url',
    ]));
    $this->setConfigurationValue('completion_event_name', $form_state->getValue([
      'completion',
      'completion_event_name',
    ]));
    $this->setConfigurationValue('completion_event_data', $form_state->getValue([
      'completion',
      'completion_event_data',
    ]));

    // Calculate and save timestamp from the new field structure.
    $date_row = $form_state->getValue([
      'target_time',
      'date_row',
    ]);
    $time_row = $form_state->getValue([
      'target_time',
      'time_row',
    ]);

    $countdown_timestamp = mktime(
      (int) $time_row['hour'],
      (int) $time_row['min'],
      (int) $time_row['sec'],
      (int) $date_row['month'],
      (int) $date_row['day'],
      (int) $date_row['year']
    );

    $this->setConfigurationValue('timestamp', $countdown_timestamp);
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state) {
    $url = $form_state->getValue(['event_settings', 'event', 'event_link']);

    // Validate URL if provided.
    if (!empty($url) && $url !== "<front>") {
      if (!UrlHelper::isExternal($url) && strpos($url, '/') !== 0) {
        $form_state->setErrorByName('event_settings][event][event_link', $this->t('Internal paths must start with a forward slash (/).'));
      }
    }

    // Only validate JavaScript settings if in realtime mode.
    $render_mode = $form_state->getValue(['display_settings', 'render_mode']);
    if ($render_mode === 'realtime') {
      // Validate custom template if custom style is selected.
      $display_style = $form_state->getValue(['js_settings', 'display_style']);
      $custom_template = $form_state->getValue(['js_settings', 'custom_template']);

      if ($display_style === 'custom' && empty($custom_template)) {
        $form_state->setErrorByName('js_settings][custom_template', $this->t('A custom template is required when custom display style is selected.'));
      }
    }

    // Validate completion settings (for both modes).
    $completion_action = $form_state->getValue(['completion', 'completion_action']);

    if ($completion_action === 'redirect') {
      $completion_url = $form_state->getValue(['completion', 'completion_url']);
      if (empty($completion_url)) {
        $form_state->setErrorByName('completion][completion_url', $this->t('A redirect URL is required when redirect action is selected.'));
      }
      // Validate URL format.
      elseif (!UrlHelper::isValid($completion_url, TRUE) && !UrlHelper::isValid($completion_url, FALSE) && $completion_url !== '<front>') {
        $form_state->setErrorByName('completion][completion_url', $this->t('Please enter a valid URL.'));
      }
    }
    elseif ($completion_action === 'message') {
      $completion_message = $form_state->getValue(['completion', 'completion_message']);
      if (empty($completion_message)) {
        $form_state->setErrorByName('completion][completion_message', $this->t('A message is required when message action is selected.'));
      }
    }
    elseif ($completion_action === 'event') {
      $event_name = $form_state->getValue(['completion', 'completion_event_name']);
      if (empty($event_name)) {
        $form_state->setErrorByName('completion][completion_event_name', $this->t('An event name is required when custom event action is selected.'));
      }

      // Validate event name format.
      if (!preg_match('/^[a-zA-Z][a-zA-Z0-9:_\-\.]*$/', $event_name)) {
        $form_state->setErrorByName('completion][completion_event_name', $this->t('Event name must start with a letter and contain only letters, numbers, colons, underscores, hyphens, and dots.'));
      }

      // Validate JSON data if provided.
      $event_data = $form_state->getValue(['completion', 'completion_event_data']);
      if (!empty($event_data)) {
        json_decode($event_data);
        if (json_last_error() !== JSON_ERROR_NONE) {
          $form_state->setErrorByName('completion][completion_event_data', $this->t('Event data must be valid JSON format. Error: @error', [
            '@error' => json_last_error_msg(),
          ]));
        }
      }
    }

    // Validate date components from the new structure.
    $date_row = $form_state->getValue(['target_time', 'date_row']);
    if (!checkdate((int) $date_row['month'], (int) $date_row['day'], (int) $date_row['year'])) {
      $form_state->setErrorByName('target_time][date_row', $this->t('The specified date is invalid.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    // Disable caching for countdown blocks.
    return 0;
  }

}
