<?php

namespace Drupal\Tests\countdown\Functional;

use Drupal\block\Entity\Block;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Countdown block functionality comprehensively.
 *
 * @group countdown
 */
class CountdownBlockTest extends BrowserTestBase {

  /**
   * Set to TRUE to strict check all configuration saved.
   *
   * @var bool
   *
   * @see \Drupal\Core\Config\Testing\ConfigSchemaChecker
   */
  protected $strictConfigSchema = TRUE;

  /**
   * An administrative user to configure the test environment.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A standard authenticated user for permission testing.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $authenticatedUser;

  /**
   * Modules to install.
   *
   * @var array
   */
  protected static $modules = ['block', 'countdown', 'node', 'field_ui'];

  /**
   * The default theme to use for testing.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create users with appropriate permissions.
    $this->adminUser = $this->drupalCreateUser([
      'administer blocks',
      'access administration pages',
    ]);

    $this->authenticatedUser = $this->drupalCreateUser([
      'access content',
    ]);

    // Log in as admin by default.
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests basic countdown block creation and configuration.
   */
  public function testCountdownBlockCreation() {
    // Navigate to block placement page.
    $default_theme = $this->config('system.theme')->get('default');
    $block_add_url = Url::fromRoute('block.admin_library', [
      'theme' => $default_theme,
    ]);
    $this->drupalGet($block_add_url);
    $this->assertSession()->statusCodeEquals(200);

    // Verify countdown block is available.
    $this->assertSession()->pageTextContains('Countdown');

    // Place a countdown block.
    $this->drupalGet('admin/structure/block/add/countdown_block/' . $default_theme);
    $this->assertSession()->statusCodeEquals(200);

    // Test form validation - submit without required fields.
    $edit = [
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');

    // Event name is no longer strictly required (backward compatibility).
    // The block should save successfully with defaults.
    $this->assertSession()->pageTextContains('The block configuration has been saved');
  }

  /**
   * Tests countdown block with static render mode.
   */
  public function testStaticRenderMode() {
    $block_id = strtolower($this->randomMachineName(8));

    // Create block with static mode configuration.
    $this->placeCountdownBlock($block_id, [
      'event_name' => 'Test Static Event',
      'event_link' => '/node/1',
      'render_mode' => 'static',
      'accuracy' => 's',
      // Tomorrow.
      'timestamp' => time() + 86400,
    ]);

    // Visit the front page and verify static countdown display.
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // Check for static countdown elements.
    $this->assertSession()->responseContains('countdown-static-content');
    $this->assertSession()->responseContains('Test Static Event');
    $this->assertSession()->responseContains('1 day');

    // Verify no JavaScript libraries are loaded for static mode.
    $this->assertSession()->responseNotContains('countdown.integration.js');
    $this->assertSession()->responseNotContains('countdown-timer-container');
  }

  /**
   * Tests countdown block with realtime JavaScript render mode.
   */
  public function testRealtimeRenderMode() {
    $block_id = strtolower($this->randomMachineName(8));

    // Create block with realtime mode configuration.
    $block = $this->placeCountdownBlock($block_id, [
      'event_name' => 'Test Realtime Event',
      'event_link' => 'https://example.com',
      'render_mode' => 'realtime',
      'timer_mode' => 'countdown',
      'precision' => 'seconds',
      'display_style' => 'verbose',
      // 1 hour from now.
      'timestamp' => time() + 3600,
    ]);

    // Visit the front page.
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // Check for realtime countdown elements and JavaScript.
    $this->assertSession()->responseContains('countdown-realtime-content');
    $this->assertSession()->responseContains('countdown-timer-container');
    $this->assertSession()->responseContains('data-block-id="' . $block->id() . '"');

    // Verify JavaScript libraries are loaded.
    $this->assertSession()->responseContains('countdown/timer');
    $this->assertSession()->responseContains('countdown/integration');

    // Check drupalSettings are properly set.
    $this->assertSession()->responseContains('drupalSettings');
    $this->assertSession()->responseContains('countdown_' . $block->id());
  }

  /**
   * Tests various display styles and formats.
   */
  public function testDisplayStyles() {
    // Test verbose style.
    $this->placeCountdownBlock('verbose_test', [
      'event_name' => 'Verbose Event',
      'render_mode' => 'realtime',
      'display_style' => 'verbose',
      'separator' => ', ',
      'show_zero' => TRUE,
      'max_units' => 3,
      // Complex time for testing.
      'timestamp' => time() + 93784,
    ]);

    // Test compact style.
    $this->placeCountdownBlock('compact_test', [
      'event_name' => 'Compact Event',
      'render_mode' => 'realtime',
      'display_style' => 'compact',
      'separator' => ' ',
      'timestamp' => time() + 7200,
    ]);

    // Test custom template.
    $this->placeCountdownBlock('custom_test', [
      'event_name' => 'Custom Template Event',
      'render_mode' => 'realtime',
      'display_style' => 'custom',
      'custom_template' => 'DD days, HH:MM:SS remaining',
      'timestamp' => time() + 172800,
    ]);

    $this->drupalGet('<front>');

    // Verify all blocks are rendered with correct settings.
    $this->assertSession()->responseContains('countdown_verbose_test');
    $this->assertSession()->responseContains('countdown_compact_test');
    $this->assertSession()->responseContains('countdown_custom_test');

    // Check JavaScript settings for each style.
    $this->assertSession()->responseContains('"display_style":"verbose"');
    $this->assertSession()->responseContains('"display_style":"compact"');
    $this->assertSession()->responseContains('"display_style":"custom"');
    $this->assertSession()->responseContains('"custom_template":"DD days, HH:MM:SS remaining"');
  }

  /**
   * Tests countdown completion actions.
   */
  public function testCompletionActions() {
    // Test completion message action.
    $this->placeCountdownBlock('message_test', [
      'event_name' => 'Message Test',
      'render_mode' => 'realtime',
      'completion_action' => 'message',
      'completion_message' => 'Event has started!',
      // Past event.
      'timestamp' => time() - 10,
    ]);

    // Test redirect action.
    $this->placeCountdownBlock('redirect_test', [
      'event_name' => 'Redirect Test',
      'render_mode' => 'realtime',
      'completion_action' => 'redirect',
      'completion_url' => '/user',
      'timestamp' => time() + 5,
    ]);

    // Test elapsed mode action.
    $this->placeCountdownBlock('elapsed_test', [
      'event_name' => 'Elapsed Test',
      'render_mode' => 'realtime',
      'completion_action' => 'elapsed',
      // Past event.
      'timestamp' => time() - 3600,
    ]);

    // Test custom event action.
    $this->placeCountdownBlock('event_test', [
      'event_name' => 'Custom Event Test',
      'render_mode' => 'realtime',
      'completion_action' => 'event',
      'completion_event_name' => 'countdown:test-complete',
      'completion_event_data' => '{"test": true, "id": 123}',
      'timestamp' => time() + 10,
    ]);

    $this->drupalGet('<front>');

    // Verify completion configurations are in drupalSettings.
    $this->assertSession()->responseContains('"completion_action":"message"');
    $this->assertSession()->responseContains('"completion_message":"Event has started!"');
    $this->assertSession()->responseContains('"completion_action":"redirect"');
    $this->assertSession()->responseContains('"completion_url":"/user"');
    $this->assertSession()->responseContains('"completion_action":"elapsed"');
    $this->assertSession()->responseContains('"completion_event_name":"countdown:test-complete"');
  }

  /**
   * Tests form validation for various fields.
   */
  public function testFormValidation() {
    $this->drupalGet('admin/structure/block/add/countdown_block/' . $this->defaultTheme);

    // Test invalid event URL.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Test Event',
      'settings[event_settings][event][event_link]' => 'invalid-url',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('Internal paths must start with a forward slash');

    // Test custom template required when custom style selected.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Test Event',
      'settings[display_settings][render_mode]' => 'realtime',
      'settings[js_settings][display_style]' => 'custom',
      'settings[js_settings][custom_template]' => '',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('A custom template is required when custom display style is selected');

    // Test completion URL validation.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Test Event',
      'settings[display_settings][render_mode]' => 'realtime',
      'settings[completion][completion_action]' => 'redirect',
      'settings[completion][completion_url]' => '',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('A redirect URL is required when redirect action is selected');

    // Test invalid JSON in event data.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Test Event',
      'settings[display_settings][render_mode]' => 'realtime',
      'settings[completion][completion_action]' => 'event',
      'settings[completion][completion_event_name]' => 'test-event',
      'settings[completion][completion_event_data]' => '{invalid json}',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('Event data must be valid JSON format');

    // Test invalid event name format.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Test Event',
      'settings[display_settings][render_mode]' => 'realtime',
      'settings[completion][completion_action]' => 'event',
      'settings[completion][completion_event_name]' => '123-invalid',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('Event name must start with a letter');
  }

  /**
   * Tests countdown timer modes (countdown vs countup).
   */
  public function testTimerModes() {
    // Test countdown mode for future event.
    $this->placeCountdownBlock('countdown_future', [
      'event_name' => 'Future Countdown',
      'render_mode' => 'realtime',
      'timer_mode' => 'countdown',
      'timestamp' => time() + 7200,
    ]);

    // Test countup mode for past event.
    $this->placeCountdownBlock('countup_past', [
      'event_name' => 'Past Countup',
      'render_mode' => 'realtime',
      'timer_mode' => 'countup',
      'timestamp' => time() - 3600,
    ]);

    $this->drupalGet('<front>');

    // Verify timer modes in JavaScript settings.
    $this->assertSession()->responseContains('"timer_mode":"countdown"');
    $this->assertSession()->responseContains('"timer_mode":"countup"');

    // Check for proper direction text.
    $this->assertSession()->responseContains('until');
    $this->assertSession()->responseContains('since');
  }

  /**
   * Tests precision settings for realtime mode.
   */
  public function testPrecisionSettings() {
    $precisions = [
      'minutes' => 'HH:MM',
      'seconds' => 'HH:MM:SS',
      'tenths' => 'HH:MM:SS.m',
      'hundredths' => 'HH:MM:SS.mm',
      'milliseconds' => 'HH:MM:SS.mmm',
    ];

    foreach ($precisions as $precision => $format) {
      $this->placeCountdownBlock('precision_' . $precision, [
        'event_name' => 'Precision Test ' . $precision,
        'render_mode' => 'realtime',
        'precision' => $precision,
        'timestamp' => time() + 3600,
      ]);
    }

    $this->drupalGet('<front>');

    // Verify all precision settings are in JavaScript config.
    foreach (array_keys($precisions) as $precision) {
      $this->assertSession()->responseContains('"precision":"' . $precision . '"');
    }
  }

  /**
   * Tests timezone configuration and display.
   */
  public function testTimezoneSettings() {
    $this->placeCountdownBlock('timezone_test', [
      'event_name' => 'Timezone Event',
      'render_mode' => 'realtime',
      'timezone' => 'America/New_York',
      'show_timezone' => TRUE,
      'timestamp' => time() + 86400,
    ]);

    $this->drupalGet('<front>');

    // Verify timezone settings in JavaScript config.
    $this->assertSession()->responseContains('"timezone":"America/New_York"');
    $this->assertSession()->responseContains('"show_timezone":true');
  }

  /**
   * Tests advanced JavaScript settings.
   */
  public function testAdvancedSettings() {
    $this->placeCountdownBlock('advanced_test', [
      'event_name' => 'Advanced Test',
      'render_mode' => 'realtime',
      'offset' => 300,
      'auto_start' => FALSE,
      'drift_compensation' => TRUE,
      'enable_events' => TRUE,
      'debug_mode' => TRUE,
      'timestamp' => time() + 3600,
    ]);

    $this->drupalGet('<front>');

    // Verify advanced settings in JavaScript config.
    $this->assertSession()->responseContains('"offset":300');
    $this->assertSession()->responseContains('"auto_start":false');
    $this->assertSession()->responseContains('"drift_compensation":true');
    $this->assertSession()->responseContains('"enable_events":true');
    $this->assertSession()->responseContains('"debug_mode":true');
  }

  /**
   * Tests backward compatibility with old block configurations.
   */
  public function testBackwardCompatibility() {
    // Create a block using old configuration structure.
    $values = [
      'id' => 'countdown_legacy',
      'plugin' => 'countdown_block',
      'region' => 'sidebar_first',
      'settings' => [
        'label' => 'Legacy Countdown',
        'label_display' => 'visible',
        // Old field names.
        'event_name' => 'Legacy Event',
        'url' => '/old-path',
        'accuracy' => 'h',
        'timestamp' => time() + 7200,
      ],
      'theme' => $this->defaultTheme,
    ];

    $block = Block::create($values);
    $block->save();

    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // Verify legacy block still renders correctly in static mode.
    $this->assertSession()->responseContains('Legacy Event');
    $this->assertSession()->responseContains('countdown-static');

    // Verify old URL field is handled.
    $this->assertSession()->responseContains('/old-path');
  }

  /**
   * Tests block visibility and permission settings.
   */
  public function testBlockVisibilityAndPermissions() {
    // Create block visible only to authenticated users.
    $this->placeCountdownBlock('auth_only', [
      'event_name' => 'Authenticated Only Event',
      'render_mode' => 'static',
      'timestamp' => time() + 3600,
    ], [
      'visibility' => [
        'user_role' => [
          'id' => 'user_role',
          'roles' => [
            'authenticated' => 'authenticated',
          ],
          'negate' => FALSE,
          'context_mapping' => [
            'user' => '@user.current_user_context:current_user',
          ],
        ],
      ],
    ]);

    // Test as authenticated user.
    $this->drupalLogin($this->authenticatedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->responseContains('Authenticated Only Event');

    // Test as anonymous user.
    $this->drupalLogout();
    $this->drupalGet('<front>');
    $this->assertSession()->responseNotContains('Authenticated Only Event');
  }

  /**
   * Tests multiple countdown blocks on the same page.
   */
  public function testMultipleBlocks() {
    // Create multiple blocks with different configurations.
    $blocks = [
      'static_block' => [
        'event_name' => 'Static Block',
        'render_mode' => 'static',
        'accuracy' => 'd',
      ],
      'realtime_verbose' => [
        'event_name' => 'Realtime Verbose',
        'render_mode' => 'realtime',
        'display_style' => 'verbose',
      ],
      'realtime_compact' => [
        'event_name' => 'Realtime Compact',
        'render_mode' => 'realtime',
        'display_style' => 'compact',
      ],
    ];

    foreach ($blocks as $id => $config) {
      $config['timestamp'] = time() + rand(3600, 86400);
      $this->placeCountdownBlock($id, $config);
    }

    $this->drupalGet('<front>');

    // Verify all blocks are rendered independently.
    $this->assertSession()->responseContains('Static Block');
    $this->assertSession()->responseContains('Realtime Verbose');
    $this->assertSession()->responseContains('Realtime Compact');

    // Check for unique block IDs.
    $this->assertSession()->responseContains('countdown_static_block');
    $this->assertSession()->responseContains('countdown_realtime_verbose');
    $this->assertSession()->responseContains('countdown_realtime_compact');
  }

  /**
   * Tests static mode completion actions.
   */
  public function testStaticCompletionActions() {
    // Test hide action for completed countdown.
    $this->placeCountdownBlock('static_hide', [
      'event_name' => 'Hide Test',
      'render_mode' => 'static',
      'completion_action' => 'hide',
      // Past event.
      'timestamp' => time() - 10,
    ]);

    // Test message action for completed countdown.
    $this->placeCountdownBlock('static_message', [
      'event_name' => 'Message Test',
      'render_mode' => 'static',
      'completion_action' => 'message',
      'completion_message' => 'Static event completed!',
      'timestamp' => time() - 10,
    ]);

    // Test elapsed action for completed countdown.
    $this->placeCountdownBlock('static_elapsed', [
      'event_name' => 'Elapsed Test',
      'render_mode' => 'static',
      'completion_action' => 'elapsed',
      'timestamp' => time() - 3600,
    ]);

    $this->drupalGet('<front>');

    // Verify hide action - block should not be visible.
    $this->assertSession()->responseNotContains('Hide Test');

    // Verify message action.
    $this->assertSession()->responseContains('Static event completed!');

    // Verify elapsed action shows time since event.
    $this->assertSession()->responseContains('since');
  }

  /**
   * Tests date/time field interaction and validation.
   */
  public function testDateTimeFields() {
    $this->drupalGet('admin/structure/block/add/countdown_block/' . $this->defaultTheme);

    // Test invalid date.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Date Test',
      'settings[target_time][date_row][month]' => '2',
      // Invalid for February.
      'settings[target_time][date_row][day]' => '31',
      'settings[target_time][date_row][year]' => '2024',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('The specified date is invalid');

    // Test valid date with all time components.
    $edit = [
      'settings[event_settings][event][event_name]' => 'Valid Date Test',
      'settings[target_time][date_row][month]' => '12',
      'settings[target_time][date_row][day]' => '25',
      'settings[target_time][date_row][year]' => '2024',
      'settings[target_time][time_row][hour]' => '14',
      'settings[target_time][time_row][min]' => '30',
      'settings[target_time][time_row][sec]' => '45',
      'region' => 'sidebar_first',
    ];
    $this->submitForm($edit, 'Save block');
    $this->assertSession()->pageTextContains('The block configuration has been saved');
  }

  /**
   * Tests external and internal URL handling.
   */
  public function testEventUrlHandling() {
    // Test external URL.
    $this->placeCountdownBlock('external_url', [
      'event_name' => 'External Event',
      'event_link' => 'https://www.example.com/event',
      'render_mode' => 'static',
      'timestamp' => time() + 3600,
    ]);

    // Test internal path.
    $this->placeCountdownBlock('internal_path', [
      'event_name' => 'Internal Event',
      'event_link' => '/user/login',
      'render_mode' => 'static',
      'timestamp' => time() + 3600,
    ]);

    // Test front page token.
    $this->placeCountdownBlock('front_page', [
      'event_name' => 'Front Page Event',
      'event_link' => '<front>',
      'render_mode' => 'static',
      'timestamp' => time() + 3600,
    ]);

    $this->drupalGet('<front>');

    // Verify all URL types are handled correctly.
    $this->assertSession()->responseContains('href="https://www.example.com/event"');
    $this->assertSession()->responseContains('href="/user/login"');
    $this->assertSession()->responseContains('href="/"');
  }

  /**
   * Helper method to place a countdown block with given configuration.
   *
   * @param string $id
   *   The block ID.
   * @param array $countdown_config
   *   Countdown-specific configuration.
   * @param array $block_config
   *   Additional block configuration.
   *
   * @return \Drupal\block\Entity\Block
   *   The created block entity.
   */
  protected function placeCountdownBlock($id, array $countdown_config = [], array $block_config = []) {
    $default_config = [
      'event_name' => 'Test Event',
      'event_link' => '',
      'timestamp' => time() + 86400,
      'render_mode' => 'static',
      'accuracy' => 's',
    ];

    $settings = array_merge($default_config, $countdown_config);

    $values = array_merge([
      'id' => 'countdown_' . $id,
      'plugin' => 'countdown_block',
      'region' => 'sidebar_first',
      'settings' => array_merge([
        'label' => 'Countdown: ' . ($settings['event_name'] ?? 'Test'),
        'label_display' => 'visible',
      ], $settings),
      'theme' => $this->defaultTheme,
    ], $block_config);

    $block = Block::create($values);
    $block->save();

    return $block;
  }

}
