<?php

namespace Drupal\countries_import\Form;

use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\countries_import\Services\CountriesService;
use Drupal\countries_import\Services\RegionsService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CountriesSettings form.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CountriesSettingsForm extends SettingsFormBase {

  const MAPPING_TITLES = [
    'name' => 'Country name',
    'code2l' => 'ISO2',
    'code3l' => 'ISO3',
    'name_official' => 'Country official name',
    'flag' => 'Flag image',
    'geographic_regions' => 'Regions',
  ];

  /**
   * The countries service manager.
   *
   * @var \Drupal\countries_import\Services\CountriesService
   */
  protected CountriesService $countriesService;

  /**
   * The regions service manager.
   *
   * @var \Drupal\countries_import\Services\RegionsService
   */
  protected RegionsService $regionsService;

  /**
   * System file configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config;

  /**
   * Creates a new CountriesSettings form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info.
   * @param \Drupal\content_translation\ContentTranslationManagerInterface $translationManager
   *   The translation manager.
   * @param \Drupal\countries_import\Services\CountriesService $countriesService
   *   The countries service manager.
   * @param \Drupal\countries_import\Services\RegionsService $regionsService
   *   The countries service manager.
   */
  public function __construct(ConfigFactoryInterface $configFactory, TypedConfigManagerInterface $typedConfigManager, EntityFieldManagerInterface $entityFieldManager, EntityTypeManagerInterface $entityTypeManager, EntityTypeBundleInfoInterface $entityTypeBundleInfo, ContentTranslationManagerInterface $translationManager, CountriesService $countriesService, RegionsService $regionsService) {
    parent::__construct($configFactory, $typedConfigManager, $entityFieldManager, $entityTypeManager, $entityTypeBundleInfo, $translationManager);
    $this->countriesService = $countriesService;
    $this->regionsService = $regionsService;
    $this->config = $this->config('countries_import.settings');
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('content_translation.manager'),
      $container->get('countries_import.helper'),
      $container->get('regions_import.helper'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'countries_import_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return ['countries_import.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @SuppressWarnings(PHPMD.CamelCaseParameterName)
   * @SuppressWarnings(PHPMD.CamelCaseVariableName)
   * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entityTypeId = $this->config->get('entity_type_id') ?: 'taxonomy_term';
    if (isset($form_state->getUserInput()['entity_type_id'])) {
      $entityTypeId = $form_state->getUserInput()['entity_type_id'];
    }
    $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
    $bundleOptions = $this->getEntityTypeBundles($entityTypeId);

    $form['#prefix'] = '<div id="countries-import-form-wrapper">';
    $form['#suffix'] = '</div>';
    $form['info'] = [
      0 => [
        '#type' => 'container',
        '#weight' => 0,
        'message' => [
          '#type' => 'markup',
          '#markup' => $this->t('The list of countries or areas contains the names of countries or areas in alphabetical order, their three-digit numerical code. <br/> By default, the name of the entity will be "Country name". You can override this configuration by choosing to save another field from below in name.'),
        ],
      ],
    ];
    $form['entity_type_id'] = [
      '#type' => 'radios',
      '#title' => $this->t('Choose storage'),
      '#description' => $this->t("Choose where to store countries"),
      '#default_value' => $entityTypeId,
      '#options' => [
        'taxonomy_term' => $this->t('Taxonomy'),
        'node' => $this->t('Content type'),
      ],
      '#return_value' => TRUE,
      '#required' => TRUE,
      '#weight' => 1,
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::updateBundlesList',
        'wrapper' => 'countries-import-form-wrapper',
        'effect' => 'fade',
        'method' => 'replaceWith',
      ],
    ];
    $form['bundle'] = [
      '#type' => 'select',
      '#title' => $entityType->getBundleLabel(),
      '#options' => array_merge(['' => '- None -'], $bundleOptions),
      '#default_value' => $this->config->get('bundle') ?: '',
      '#required' => TRUE,
      '#size' => 1,
      '#multiple' => FALSE,
      '#weight' => 2,
      '#ajax' => [
        'callback' => '::updateEntityListItems',
        'wrapper' => 'countries-import-form-wrapper',
        'effect' => 'fade',
        'method' => 'replaceWith',
      ],
    ];
    $this->setFormFields($form, $form_state, $entityTypeId);
    $form['flag_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Import flags'),
      '#options' => [
        'svg' => $this->t('SVG format'),
        'png-32' => $this->t('PNG format 32x16'),
        'png-128' => $this->t('PNG format 128x64'),
      ],
      '#default_value' => $this->config->get('flag_format') ?: 'svg',
      '#required' => FALSE,
      '#size' => 1,
      '#multiple' => FALSE,
      '#weight' => 10,
      '#states' => [
        'invisible' => [
          ':input[name="flag"]' => ['value' => ''],
        ],
      ],
    ];
    $form['translate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Translate name and official name'),
      '#default_value' => $form_state->getUserInput()['translate'] ?? $this->config->get('translate'),
      '#weight' => 15,
      '#states' => [
        'invisible' => [
          ':input[name="bundle"]' => ['value' => ''],
        ],
      ],
    ];
    $this->setRegionsFields($form, $form_state);
    $form['actions']['import'] = [
      '#type' => 'submit',
      '#value' => $this->t('Import Countries'),
      '#weight' => 15,
      '#submit' => [
        [$this, 'submitForm'],
        [$this, 'importCountriesForm'],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Set fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state array.
   * @param string $entityTypeId
   *   The entity type ID. Only entity types that implement
   *   \Drupal\Core\Entity\FieldableEntityInterface are supported.
   */
  protected function setFormFields(array &$form, FormStateInterface $formState, string $entityTypeId) {
    foreach (self::MAPPING_TITLES as $fieldConfigId => $fieldTitle) {
      $this->setFormField($form, $formState, $entityTypeId, $fieldConfigId, $fieldTitle);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getRegionsSettings() {
    return $this->config('countries_import.geographic_regions.settings');
  }

  /**
   * Set region field and settings.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state array.
   */
  protected function setRegionsFields(&$form, $formState) {
    $regionsSettings = $this->getRegionsSettings();
    $message = new TranslatableMarkup('This option is available after you configure <a href=":url">Geographic Regions</a> settings.', [
      ':url' => Url::fromRoute('geographic_regions.settings')->toString(),
    ]);

    $form['regions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Import the geographic regions and the assignment of countries to regions.'),
      '#default_value' => $formState->getUserInput()['regions'] ?? $this->config->get('regions'),
      '#weight' => 12,
      '#disabled' => empty($regionsSettings->get('fields')),
      '#description' => $message,
      '#states' => [
        'invisible' => [
          ':input[name="bundle"]' => ['value' => ''],
        ],
      ],
    ];

    if (isset($form['geographic_regions'])) {
      $form['geographic_regions']['#description'] = $this->t('Each country or area is shown in
<a href="https://unstats.un.org/unsd/methodology/m49" target="_blank">a specific region</a>.');
      $form['geographic_regions']['#disabled'] = empty($regionsSettings->get('fields'));
      $form['geographic_regions']['#states']['disabled'][] = [
        ':input[name="regions"]' => ['checked' => FALSE],
      ];
    }
  }

  /**
   * Creates entities using the configuration.
   *
   * @throws \Exception
   */
  public function importCountriesForm() {
    if ($this->config->get('regions')) {
      try {
        $this->regionsService->import();
        $this->messenger()->addStatus($this->t('Geographic regions was imported successfully.'));
      }
      catch (\Exception $e) {
        throw new \Exception($e);
      }
    }
    try {
      $this->countriesService->import();
      $this->messenger()->addStatus($this->t('Countries was imported successfully.'));
    }
    catch (\Exception $e) {
      throw new \Exception($e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $formState) {
    $values = $formState->getValues();
    if ($values['translate'] && !$this->translationManager->isEnabled($values['entity_type_id'], $values['bundle'])) {
      $message = $this->t('Translation is not enable for this type of entity.
      Uncheck "Translate name and official name" or enable translation.');
      $formState->setErrorByName('translate', $message);
    }
    // Change regions to FALSE if is TRUE but settings for regions are not set.
    if ($values['regions'] && !$this->getRegionsSettings()->get('fields')) {
      $formState->setValue('regions', 0);
      $this->config->set('regions', 0);
      $this->config->save();
    }
    parent::validateForm($form, $formState);
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function submitForm(array &$form, FormStateInterface $formState) {
    parent::submitForm($form, $formState);

    $values = $formState->getValues();
    foreach (array_keys(self::MAPPING_TITLES) as $fieldId) {
      $value = $values[$fieldId] ?? '';
      if (!in_array($value, $form[$fieldId]['#options'])) {
        $value = '';
      }
      $this->config->set('fields.' . $fieldId, $value);
    }
    $this->config->set('flag_format', $values['flag_format']);
    // If true, import regions and relations.
    $this->config->set('regions', $values['regions']);
    // If TRUE, import translations.
    $this->config->set('translate', $values['translate']);
    $this->config->save();

  }

  /**
   * {@inheritdoc}
   */
  public function updateEntityListItems(array &$form): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpUnused
   */
  public function updateBundlesList(array &$form): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedFieldType($field) {
    switch ($field) {
      case 'name':
      case 'code2l':
      case 'code3l':
      case 'name_official':
        return [
          'string',
          'string_long',
          'text',
          'text_long',
          'text_with_summary',
        ];

      case 'flag':
        return [
          'file',
          'image',
          'media',
        ];

      case 'geographic_regions':
        return [
          $this->getRegionsSettings()->get('entity_type_id'),
        ];
    }
  }

}
