<?php

namespace Drupal\countries_import\Form;

use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\countries_import\Services\RegionsService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * RegionsSettings form.
 */
class RegionsSettingsForm extends SettingsFormBase {

  const MAPPING_TITLES = [
    'name' => 'Region name',
    'id' => 'M49 code',
  ];

  /**
   * The regions service manager.
   *
   * @var \Drupal\countries_import\Services\RegionsService
   */
  protected RegionsService $regionsService;

  /**
   * System file configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config;

  /**
   * Creates a new RegionSettings form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info.
   * @param \Drupal\content_translation\ContentTranslationManagerInterface $translationManager
   *   The translation manager.
   * @param \Drupal\countries_import\Services\RegionsService $regionsService
   *   The regions service manager.
   */
  public function __construct(ConfigFactoryInterface $configFactory, TypedConfigManagerInterface $typedConfigManager, EntityFieldManagerInterface $entityFieldManager, EntityTypeManagerInterface $entityTypeManager, EntityTypeBundleInfoInterface $entityTypeBundleInfo, ContentTranslationManagerInterface $translationManager, RegionsService $regionsService) {
    parent::__construct($configFactory, $typedConfigManager, $entityFieldManager, $entityTypeManager, $entityTypeBundleInfo, $translationManager);
    $this->regionsService = $regionsService;
    $this->config = $this->config('countries_import.geographic_regions.settings');
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('content_translation.manager'),
      $container->get('regions_import.helper'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'geographic_regions_import_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return ['countries_import.geographic_regions.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @SuppressWarnings(PHPMD.CamelCaseParameterName)
   * @SuppressWarnings(PHPMD.CamelCaseVariableName)
   * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entityTypeId = $this->config->get('entity_type_id') ?: 'taxonomy_term';
    if (isset($form_state->getUserInput()['entity_type_id'])) {
      $entityTypeId = $form_state->getUserInput()['entity_type_id'];
    }
    $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
    $bundleOptions = $this->getEntityTypeBundles($entityTypeId);

    $form['#prefix'] = '<div id="regions-import-form-wrapper">';
    $form['#suffix'] = '</div>';
    $form['info'] = [
      0 => [
        '#type' => 'container',
        '#weight' => 0,
        'message' => [
          '#type' => 'markup',
          '#markup' => $this->t('The list of geographic regions presents the 
composition of geographical regions used by the Statistics Division in its 
publications and databases. Each country or area is shown in one region only. 
These geographic regions are based on continental regions; which are further 
subdivided into sub-regions and intermediary regions drawn as to obtain greater 
homogeneity in sizes of population, demographic circumstances and accuracy of 
demographic statistics.'),
        ],
      ],
    ];
    $form['entity_type_id'] = [
      '#type' => 'radios',
      '#title' => $this->t('Choose storage'),
      '#description' => $this->t("Choose where to store regions"),
      '#default_value' => $entityTypeId,
      '#options' => [
        'taxonomy_term' => $this->t('Taxonomy'),
        'node' => $this->t('Content type'),
      ],
      '#return_value' => TRUE,
      '#required' => TRUE,
      '#weight' => 1,
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::updateBundlesList',
        'wrapper' => 'regions-import-form-wrapper',
        'effect' => 'fade',
        'method' => 'replaceWith',
      ],
    ];
    $form['bundle'] = [
      '#type' => 'select',
      '#title' => $entityType->getBundleLabel(),
      '#options' => array_merge(['' => '- None -'], $bundleOptions),
      '#default_value' => $this->config->get('bundle') ?: '',
      '#required' => TRUE,
      '#size' => 1,
      '#multiple' => FALSE,
      '#weight' => 2,
      '#ajax' => [
        'callback' => '::updateEntityListItems',
        'wrapper' => 'regions-import-form-wrapper',
        'effect' => 'fade',
        'method' => 'replaceWith',
      ],
    ];
    $this->setFormFields($form, $form_state, $entityTypeId);
    $form['translate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Translate name and official name'),
      '#default_value' => $form_state->getUserInput()['translate'] ?? $this->config->get('translate'),
      '#weight' => 11,
      '#states' => [
        'invisible' => [
          ':input[name="bundle"]' => ['value' => ''],
        ],
      ],
    ];
    $form['actions']['import'] = [
      '#type' => 'submit',
      '#value' => $this->t('Import geographical regions'),
      '#weight' => 11,
      '#submit' => [
        [$this, 'submitForm'],
        [$this, 'importRegionsForm'],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Set fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state array.
   * @param string $entityTypeId
   *   The entity type ID. Only entity types that implement
   *   \Drupal\Core\Entity\FieldableEntityInterface are supported.
   */
  protected function setFormFields(array &$form, FormStateInterface $formState, string $entityTypeId) {
    foreach (self::MAPPING_TITLES as $fieldConfigId => $fieldTitle) {
      $this->setFormField($form, $formState, $entityTypeId, $fieldConfigId, $fieldTitle);
    }
    $message = new TranslatableMarkup('UN M49 or the Standard Country is a standard
     for area codes used by the United Nations for statistical purposes, 
     developed and maintained by the United Nations Statistics Division.
     <a href=":url" target="_blank">See more here.</a>', [
       ':url' => 'https://unstats.un.org/unsd/methodology/m49',
     ]);
    $form['id']['#description'] = $message;
  }

  /**
   * Creates entities using the configuration.
   *
   * @throws \Exception
   */
  public function importRegionsForm() {
    try {
      $this->regionsService->import();
      $this->messenger()->addStatus($this->t('Geographic regions was imported successfully.'));
    }
    catch (\Exception $e) {
      throw new \Exception($e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $formState) {
    $values = $formState->getValues();
    if ($values['translate'] && !$this->translationManager->isEnabled($values['entity_type_id'], $values['bundle'])) {
      $message = $this->t('Translation is not enable for this type of entity.
      Uncheck "Translate name and official name" or enable translation.');
      $formState->setErrorByName('translate', $message);
    }
    parent::validateForm($form, $formState);
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function submitForm(array &$form, FormStateInterface $formState) {
    parent::submitForm($form, $formState);

    $values = $formState->getValues();
    foreach (array_keys(self::MAPPING_TITLES) as $fieldId) {
      $value = $values[$fieldId] ?? '';
      if (!in_array($value, $form[$fieldId]['#options'])) {
        $value = '';
      }
      $this->config->set('fields.' . $fieldId, $value);
    }
    $this->config->set('translate', $values['translate']);
    $this->config->save();
  }

  /**
   * {@inheritdoc}
   */
  public function updateEntityListItems(array &$form): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpUnused
   */
  public function updateBundlesList(array &$form): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedFieldType($field) {
    switch ($field) {
      case 'name':
      case 'id':
        return [
          'string',
          'string_long',
          'text',
          'text_long',
          'text_with_summary',
        ];

      default:
        return [];
    }
  }

}
