<?php

namespace Drupal\countries_import\Form;

use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CountriesSettings form.
 */
class SettingsFormBase extends ConfigFormBase {

  const MAPPING_TITLES = [];

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The translation manager.
   *
   * @var \Drupal\content_translation\ContentTranslationManagerInterface
   */
  protected ContentTranslationManagerInterface $translationManager;

  /**
   * System file configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config;

  /**
   * Creates a new CountriesSettings form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info.
   * @param \Drupal\content_translation\ContentTranslationManagerInterface $translationManager
   *   The translation manager.
   */
  public function __construct(ConfigFactoryInterface $configFactory, TypedConfigManagerInterface $typedConfigManager, EntityFieldManagerInterface $entityFieldManager, EntityTypeManagerInterface $entityTypeManager, EntityTypeBundleInfoInterface $entityTypeBundleInfo, ContentTranslationManagerInterface $translationManager) {
    parent::__construct($configFactory, $typedConfigManager);
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
    $this->entityTypeBundleInfo = $entityTypeBundleInfo;
    $this->translationManager = $translationManager;
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('content_translation.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'base_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    return parent::buildForm($form, $form_state);
  }

  /**
   * Set fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state array.
   * @param string $entityTypeId
   *   The entity type ID. Only entity types that implement
   *   \Drupal\Core\Entity\FieldableEntityInterface are supported.
   */
  protected function setFormFields(array &$form, FormStateInterface $formState, string $entityTypeId) {
    foreach (self::MAPPING_TITLES as $fieldConfigId => $fieldTitle) {
      $this->setFormField($form, $formState, $entityTypeId, $fieldConfigId, $fieldTitle);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function setFormField(array &$form, FormStateInterface $formState, $entityTypeId, $fieldConfigId, $fieldTitle) {
    $bundle = $formState->getUserInput()['bundle'] ?? $this->config->get('bundle');
    $existingFields = $this->getEntityBundleFields($entityTypeId, $bundle, $fieldConfigId);
    $defaultValue = $formState->getUserInput()[$fieldConfigId] ?? $this->config->get('fields.' . $fieldConfigId) ?: '';
    $defaultValue = in_array($defaultValue, $existingFields) ? $defaultValue : '';

    $form[$fieldConfigId] = [
      '#type' => 'select',
      '#title' => $fieldTitle,
      '#options' => $existingFields,
      '#default_value' => $defaultValue,
      '#required' => (in_array($fieldConfigId, ['code2l', 'code3l', 'id'])),
      '#size' => 1,
      '#multiple' => FALSE,
      '#weight' => 5,
      '#validated' => TRUE,
      '#states' => [
        'invisible' => [
          ':input[name="bundle"]' => ['value' => ''],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $formState) {
    $values = $formState->getValues();
    if ($values['translate'] && !$this->translationManager->isEnabled($values['entity_type_id'], $values['bundle'])) {
      $message = $this->t('Translation is not enable for this type of entity.
      Uncheck "Translate name and official name" or enable translation.');
      $formState->setErrorByName('translate', $message);
    }
    parent::validateForm($form, $formState);
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function submitForm(array &$form, FormStateInterface $formState) {
    $values = $formState->getValues();

    $this->config->set('bundle', $values['bundle']);
    $this->config->set('entity_type_id', $values['entity_type_id']);
    $this->config->save();

    parent::submitForm($form, $formState);
  }

  /**
   * {@inheritdoc}
   */
  public function updateEntityListItems(array &$form): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @noinspection PhpUnused
   */
  public function updateBundlesList(array &$form): array {
    return $form;
  }

  /**
   * Get an array with fields from a specific bundle.
   *
   * @param string $entityTypeId
   *   The entity type ID. Only entity types that implement.
   * @param string $bundle
   *   The bundle.
   * @param string $fieldConfigId
   *   The field config id.
   *
   * @return array|string[]
   *   Return an array with values.
   */
  protected function getEntityBundleFields($entityTypeId, $bundle, $fieldConfigId): array {
    $fields = $this->entityFieldManager->getFieldDefinitions($entityTypeId, $bundle);
    foreach ($fields as $fieldId => $fieldInfo) {
      $type = ($fieldInfo->getType() == 'entity_reference') ? $fieldInfo->getSetting('target_type') : $fieldInfo->getType();
      if (!in_array($type, $this->getAllowedFieldType($fieldConfigId))) {
        unset($fields[$fieldId]);
      }
    }
    return array_merge([NULL => '- None - '], array_combine(array_keys($fields), array_keys($fields)));
  }

  /**
   * Gets the bundle info of an entity type.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   *
   * @return array
   *   An array of bundle information.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getEntityTypeBundles($entityTypeId): array {
    $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entityTypeId);
    $bundleOptions = [];
    if ($entityType->hasKey('bundle')) {
      foreach ($bundles as $bundleName => $bundleInfo) {
        $bundleOptions[$bundleName] = $bundleInfo['label'];
      }
      natsort($bundleOptions);
    }
    return $bundleOptions;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedFieldType($field) {
    switch ($field) {
      case 'name':
        return [
          'string',
          'string_long',
          'text',
          'text_long',
          'text_with_summary',
        ];

      default:
        return [];
    }
  }

}
