<?php

namespace Drupal\countries_import\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Provides a base class with common utility methods.
 */
abstract class BaseService {
  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $settings;

  /**
   * Constructor for BaseService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user interface.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $configFactory, AccountProxyInterface $currentUser, EntityTypeManagerInterface $entityTypeManager) {
    $this->configFactory = $configFactory;
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritDoc}
   */
  protected function setConfig(string $configId) {
    $this->settings = $this->configFactory->get($configId);
  }

  /**
   * {@inheritDoc}
   */
  protected function getConfig(string $configId) {
    return $this->configFactory->get($configId);
  }

  /**
   * {@inheritDoc}
   */
  public function getSetting(string $key) {
    return $this->settings->get($key);
  }

  /**
   * Create a new entity.
   *
   * @param string $name
   *   The name of the entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The new entity.
   */
  protected function createInstance(string $name): EntityInterface {
    if ($this->getSetting('entity_type_id') == 'taxonomy_term') {
      return $this->entityTypeManager->getStorage('taxonomy_term')->create([
        'vid' => $this->getSetting('bundle'),
        'name' => $name,
      ]);
    }
    $entity = $this->entityTypeManager->getStorage('node')->create([
      'type' => $this->getSetting('bundle'),
      'title' => $name,
    ]);
    $entity->setOwnerId($this->currentUser->id());
    return $entity;
  }

}
