<?php

namespace Drupal\countries_import\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\migrate\Exception\EntityValidationException;

/**
 * The RegionService service.
 */
class RegionsService extends BaseService {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Json with regions.
   *
   * @var string
   */
  protected string $path;

  /**
   * RegionService constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user interface.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   */
  public function __construct(ConfigFactoryInterface $configFactory, AccountProxyInterface $currentUser, EntityTypeManagerInterface $entityTypeManager, LanguageManagerInterface $languageManager) {
    parent::__construct($configFactory, $currentUser, $entityTypeManager);
    $this->setConfig('countries_import.geographic_regions.settings');
    $this->languageManager = $languageManager;
    $this->path = "https://raw.githubusercontent.com/cristiroma/countries/master/data/regions-m49.json";
  }

  /**
   * {@inheritdoc}
   */
  public function import() {
    $data = file_get_contents($this->path);
    $region = json_decode($data, TRUE);
    $this->importRegion([$region], '0');
  }

  /**
   * Import each region with parent & translation if needed.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\migrate\Exception\EntityValidationException
   */
  protected function importRegion(array $regions, string $parent = '0') {
    foreach ($regions as $region) {
      $properties = [
        $this->getSetting('fields')['id'] => $region['id'],
      ];
      $entity = $this->entityTypeManager->getStorage($this->getSetting('entity_type_id'))
        ->loadByProperties($properties);
      $entity = reset($entity);
      if (empty($entity)) {
        $entity = $this->createInstance($region['name']);
      }
      $entity = $this->populateFields($entity, $region);
      $errors = $entity->validate();
      if ($errors->count()) {
        throw new EntityValidationException($errors);
      }
      if ($this->getSetting('translate')) {
        $this->translateEntity($entity, $region);
      }
      $entity->set('parent', $parent);
      $entity->save();
      if (!empty($region['subregions'])) {
        $this->importRegion($region['subregions'], $entity->id());
      }
    }
  }

  /**
   * Populate entity's fields.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object being populated.
   * @param array $item
   *   Array with information.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The entity.
   */
  private function populateFields(EntityInterface $entity, array $item): EntityInterface {
    foreach ($this->getSetting('fields') as $key => $field) {
      if (!$field) {
        continue;
      }
      $entity->set($field, $item[$key]);
    }
    return $entity;
  }

  /**
   * Add translations for entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object being translated.
   * @param array $item
   *   Array with information.
   */
  private function translateEntity(EntityInterface $entity, array $item) {
    $languages = $this->languageManager->getLanguages();
    unset($languages[$this->languageManager->getDefaultLanguage()->getId()]);

    foreach ($languages as $language) {
      $langcode = $language->getId();
      if (empty($item["name_$langcode"])) {
        continue;
      }
      $translateEntity = [];
      $hasTranslation = $entity->hasTranslation($langcode);
      $fieldMachineName = $this->getSetting('fields')['name'];
      if (!$fieldMachineName) {
        continue;
      }
      if ($hasTranslation) {
        $entity->getTranslation($langcode)->$fieldMachineName = $item["name_$langcode"];
        continue;
      }
      $translateEntity[$fieldMachineName] = $item["name_$langcode"];
      if (!$hasTranslation) {
        $entity->addTranslation($langcode, $translateEntity);
      }
    }
  }

}
