<?php

namespace Drupal\country_path;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\domain\DomainElementManagerInterface;
use Drupal\domain\DomainListBuilder;
use Drupal\domain\DomainStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Override the Domain entities list to show their Country path values.
 */
class CountryPathDomainListBuilder extends DomainListBuilder {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('current_user'),
      $container->get('redirect.destination'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('domain.element_manager'),
      $container->get('request_stack')
    );
  }

  /**
   * Constructs a new CountryPathDomainListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\domain\DomainStorageInterface $domain_storage
   *   The domain storage class.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The active user account.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $destination_handler
   *   The redirect destination helper.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\domain\DomainElementManagerInterface $domain_element_manager
   *   The domain field element manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   */
  public function __construct(EntityTypeInterface $entity_type, DomainStorageInterface $domain_storage, AccountInterface $account, RedirectDestinationInterface $destination_handler, EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler, DomainElementManagerInterface $domain_element_manager, RequestStack $request_stack) {
    parent::__construct($entity_type, $domain_storage, $account, $destination_handler, $entity_type_manager, $module_handler, $domain_element_manager);
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    // Call parent's buildRow method.
    $row = parent::buildRow($entity);

    $options = ['absolute' => TRUE, 'https' => $entity->isHttps()];
    $domain_suffix = $entity->getThirdPartySetting('country_path', 'domain_path');
    $domain_path = $entity->getPath();
    $current_path = $this->requestStack->getCurrentRequest()->getPathInfo();

    if (empty($domain_suffix)) {
      $uri = $domain_path . ltrim($current_path, '/');
    }
    else {
      $uri = $domain_path . $domain_suffix . $current_path;
      $domain_suffix = "/$domain_suffix";
    }

    $hostname = Link::fromTextAndUrl(
      $entity->getCanonical() . $domain_suffix, Url::fromUri($uri, $options)
    )->toString();
    $row['hostname']['#markup'] = $hostname;
    return $row;
  }

}
