<?php

declare(strict_types=1);

namespace Drupal\coveo;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Url;
use Drupal\coveo\Entity\CoveoOrganizationInterface;

/**
 * Defines a class to build a listing of Coveo organization entities.
 *
 * @see \Drupal\coveo\Entity\CoveoOrganizationInterface
 */
class OrganizationListBuilder extends ConfigEntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('Organization');
    $header['org_id'] = $this->t('Organization ID');
    $header['read_only'] = $this->t('Read only');
    $header['prefix'] = $this->t('Prefix');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    assert($entity instanceof CoveoOrganizationInterface);
    $row['label'] = $entity->label();
    $row['org_id'] = $entity->getOrganizationId();
    $row['read_only'] = $entity->isReadOnly() ? $this->t('Read only') : '';
    $row['prefix'] = $entity->getPrefix();
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();
    $build['table']['#empty'] = $this->t('There are currently no Coveo organizations defined. <a href=":url">Add a new one</a>.', [
      ':url' => Url::fromRoute('coveo.coveo_organization_add')->toString(),
    ]);
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations(EntityInterface $entity) {
    assert($entity instanceof CoveoOrganizationInterface);
    $operations = parent::getOperations($entity);
    if (!$entity->isReadOnly()) {
      $operations['sync'] = [
        'title' => $this->t('Sync Fields'),
        'weight' => 10,
        'url' => Url::fromRoute('entity.coveo_organization.sync', [
          'coveo_organization' => $entity->id(),
        ]),
      ];
    }

    return $operations;
  }

}
