<?php

declare(strict_types=1);

namespace Drupal\coveo_secured_search\Form\SecurityProviders;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\coveo\Entity\CoveoOrganizationInterface;
use Drupal\coveo_secured_search\Plugin\CoveoCustomSecurityProviderManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base form for Coveo Security Provider add and edit forms.
 */
class CustomSecurityProviderForm extends EntityForm {

  /**
   * The entity being used by this form.
   *
   * @var \Drupal\coveo_secured_search\Entity\CoveoCustomSecurityProvider
   */
  protected $entity;

  /**
   * Constructs a base class for Coveo search add and edit forms.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $securityProviderStorage
   *   The Coveo search entity storage.
   * @param \Drupal\coveo_secured_search\Plugin\CoveoCustomSecurityProviderManagerInterface $coveoCustomSecurityProviderManager
   *   The Coveo custom security provider manager.
   */
  public function __construct(
    protected readonly EntityStorageInterface $securityProviderStorage,
    protected readonly CoveoCustomSecurityProviderManagerInterface $coveoCustomSecurityProviderManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('coveo_custom_security_provider'),
      $container->get('plugin.manager.coveo_custom_security_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Coveo Security Provider name'),
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];
    $form['name'] = [
      '#type' => 'machine_name',
      '#machine_name' => [
        'exists' => [$this->securityProviderStorage, 'load'],
      ],
      '#default_value' => $this->entity->id(),
      '#required' => TRUE,
    ];
    $form['coveo_name'] = [
      '#title' => $this->t('Coveo machine name'),
      '#type' => 'textfield',
      '#default_value' => $this->entity->getCoveoName(),
      '#description' => $this->t('Machine name used by Coveo to identity this provider. Defaults to the Drupal machine name.'),
      '#required' => FALSE,
    ];

    $form['description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Description'),
      '#default_value' => $this->entity->getDescription(),
      '#description' => $this->t('Local description of the security provider.'),
    ];
    $form['type'] = [
      '#type' => 'details',
      '#title' => $this->t('Custom security provider type'),
      '#open' => TRUE,
    ];

    $form['type']['security_provider'] = [
      '#type' => 'select',
      '#title' => 'Provider type',
      '#default_value' => $this->entity->getSecurityProviderPluginId(),
      '#options' => array_map(fn($definition) => $definition['title'], $this->coveoCustomSecurityProviderManager->getDefinitions()),
      '#required' => TRUE,
      '#description' => 'The type of provider that will be used to generate tokens and describe identities to coveo.',
    ];
    $form['organization_name'] = [
      '#type' => 'select',
      '#title' => 'Organization',
      '#default_value' => $this->entity->getOrganizationName(),
      '#options' => array_map(
        fn(CoveoOrganizationInterface $org) => $org->label(),
        $this->getOrganizations()
      ),
      '#required' => TRUE,
    ];
    // Search API enabled, link to push sources so identities can be pushed.
    if ($this->moduleHandler->moduleExists('coveo_search_api')) {
      // @todo link this to SearchApi to help creation.
      $form['push_sources'] = [
        '#markup' => $this->t('<a href=":search_api">Add a push source</a> to populate identities.', [
          ':search_api' => Url::fromRoute('search_api.overview')->toString(),
        ]),
      ];
    }

    return parent::form($form, $form_state);
  }

  /**
   * Retrieves all available Coveo organizations.
   *
   * @return \Drupal\coveo\Entity\CoveoOrganizationInterface[]
   *   The available organizations.
   */
  private function getOrganizations(): array {
    try {
      return $this->entityTypeManager
        ->getStorage('coveo_organization')
        ->loadMultiple();
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException) {
      // This should never happen.
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $return = parent::save($form, $form_state);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $return;
  }

}
