<?php

declare(strict_types=1);

namespace Drupal\coveo_secured_search\Plugin\Coveo\CustomSecurityProvider;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\coveo\Plugin\Coveo\SecurityProvider\AbstractSecuredUserProvider;
use Drupal\coveo_secured_search\Attribute\CoveoCustomSecurityProvider;
use Drupal\coveo_secured_search\Plugin\CoveoCustomSecurityProviderPluginInterface;
use Drupal\search_api\Utility\Utility;

/**
 * Drupal Provider security provider.
 */
#[CoveoCustomSecurityProvider(
  id: 'drupal_provider',
  baseName: 'DrupalProvider',
  title: new TranslatableMarkup('Drupal Provider'),
  description: new TranslatableMarkup('Custom security provider that uses the Drupal User ID as the user identifier.'),
)]
class DrupalProvider extends AbstractSecuredUserProvider implements CoveoCustomSecurityProviderPluginInterface {

  /**
   * {@inheritDoc}
   */
  #[\Override]
  public function getIdentityProviderId(): string {
    return $this->configuration['provider_id'] ?? 'Default Drupal Provider';
  }

  /**
   * {@inheritDoc}
   */
  #[\Override]
  protected function getName(AccountInterface $account): string {
    return (string) $account->id();
  }

  /**
   * {@inheritDoc}
   */
  #[\Override]
  public function getNameFromId(string $id): string {
    // Expected uri is something like 'entity:user/1:en'. Split the id.
    [, $new_id] = Utility::splitCombinedId($id);

    // Now it's formatted like 1:en, but we need to match ::getName.
    return $this->hackIdFromKey($new_id);
  }

  /**
   * Hack real ID from item key.
   *
   * HACK: Assume content entity data source or similar and remove language
   * code and return only the ID.
   *
   * @param string $key
   *   Search API key.
   *
   * @return string
   *   User id.
   *
   * @see \Drupal\search_api\Plugin\search_api\datasource\ContentEntity::formatItemId
   */
  private function hackIdFromKey(string $key): string {
    if (str_contains($key, ':')) {
      [$tmp] = explode(':', $key, 2);
      return $tmp;
    }
    return $key;
  }

}
