<?php

declare(strict_types=1);

namespace Drupal\coveo\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\coveo\Coveo\Index;
use Drupal\coveo\FieldConverter;
use NecLimDul\Coveo\SecurityCache\Api\SecurityProvidersApi;

/**
 * Interface for Coveo Organization configuration entity.
 */
interface CoveoOrganizationInterface extends ConfigEntityInterface {

  /**
   * Get the Coveo Organization ID.
   */
  public function getOrganizationId(): string|null;

  /**
   * Get a Coveo Security Provider API instance.
   *
   * @return \NecLimDul\Coveo\SecurityCache\Api\SecurityProvidersApi
   *   The SecurityProvidersApi instance.
   */
  public function getSecurityProviderApi(): SecurityProvidersApi;

  /**
   * True if the organization is to be treated as read only.
   */
  public function isReadOnly(): bool;

  /**
   * True if things like fields should auto sync with coveo.
   */
  public function autoSync(): bool;

  /**
   * Environment prefix.
   */
  public function getPrefix(): string;

  /**
   * Get the push key.
   */
  public function getPushKey(): string|null;

  /**
   * Get the push source id.
   */
  public function getPushSourceId(): string|null;

  /**
   * Get a field indexer helper.
   *
   * @return \Drupal\coveo\FieldConverter
   *   Coveo field indexer.
   */
  public function getFieldConverter(): FieldConverter;

  /**
   * Create a FieldApi service.
   *
   * @param class-string<T> $class
   *   A FieldApi service class.
   *
   * @return T
   *   The instantiated API service.
   *
   * @template T
   */
  public function fieldApiCreate(string $class): mixed;

  /**
   * Create a PushAPI service.
   *
   * @param class-string<T> $class
   *   A PushAPI service class.
   *
   * @return T
   *   The instantiated API service.
   *
   * @template T
   */
  public function pushApiCreate(string $class): mixed;

  /**
   * Create a SecurityCache API service.
   *
   * @param class-string<T> $class
   *   A SecurityCache API service class.
   *
   * @return T
   *   The instantiated API service.
   *
   * @template T
   */
  public function securityCacheCreate(string $class): mixed;

  /**
   * Get an indexing helper.
   *
   * @return \Drupal\coveo\Coveo\Index
   *   Coveo Index helper service instance.
   */
  public function getIndexHelper(): Index;

  /**
   * Sync related fields to Coveo.
   */
  public function sync(): void;

}
